/***************************************************************************
 *   Copyright (C) 2023 by Theodore H. Kaskalis                            *
 *                                                                         *
 ***( see copyright.txt file at root folder )*******************************/

#ifndef KEYBOARD_H
#define KEYBOARD_H

#include <QPainter>
#include <QWidget>
#include <QGraphicsProxyWidget>

#include "e-element.h"
#include "iopin.h"
#include "circuit.h"
#include "custombutton.h"

#define KEYBOARD_BUTTON_SIZE 101
#define KEYBOARD_BASE_STEPS 1e6*1e6
#define KEYBOARD_10_usec ( KEYBOARD_BASE_STEPS / 1e5 )
#define KEYBOARD_50_usec ( KEYBOARD_BASE_STEPS / 2e4 )
#define KEYBOARD_500_msec ( KEYBOARD_BASE_STEPS / 2.0 )
#define KEYBOARD_FREQUENCY 12500.0 // 80 usec period
// #define KEYBOARD_FREQUENCY 0.125
// #define KEYBOARD_FREQUENCY 1.25
#define KEYBOARD_DEFAULT_TYPEMATIC_DELAY 0.5 // 500 msec
#define KEYBOARD_DEFAULT_TYPEMATIC_RATE 10.9 // 10.9 characters per sec

#define CodeErrorOrOverflow 0
#define CodeKeyboardID      1
#define CodeBATPassed       2
#define CodeScanCodeSet2    3
#define CodeEcho            4
#define CodeAcknowledge     5
#define CodeResend          6

class LibraryItem;

enum DeviceStates {
    KbdDisabled          =  0,
    BATinProcess         =  1, // Un-interruptable
    HostSendsData        =  2, // Command or Argument
    DeviceHalted         =  3, // Communication Inhibited or waiting for host command
    KbdEnabled           =  4,
    DeviceSendsData      =  5,
    KbdIdle              =  6
};

enum ChunkType {
    EmptyData     = 0,
    Answer        = 1,
    BufferData    = 2,
    TypematicData = 3
};

enum Phases {
    firstHighQuarter    =  0,
    secondHighQuarter   =  1,
    firstLowQuarter     =  2,
    secondLowQuarter    =  3
};

struct KeyCode {
    int size;
    unsigned short int data[8];
};

struct Chunk {
    ChunkType type;
    KeyCode * code;
};

enum HostCommandSet {
    ComEmpty                        = 0x00,
    ComReset                        = 0xFF,
    ComResend                       = 0xFE,
    // ComSetKeyTypeMake               = 0xFD, |
    // ComSetKeyTypeMakeBreak          = 0xFC, |
    // ComSetKeyTypeTypematic          = 0xFB, |
    // ComSetAllKeysTypematicMakeBreak = 0xFA, | Scan Code Set 3 Commands NOT Supported
    // ComSetAllKeysMake               = 0xF9, |
    // ComSetAllKeysMakeBreak          = 0xF8, |
    // ComSetAllKeysTypematic          = 0xF7, |
    ComSetDefault                   = 0xF6,
    ComDisable                      = 0xF5,
    ComEnable                       = 0xF4,
    ComSetTypematicRateDelay        = 0xF3,
    ComReadID                       = 0xF2,
    ComSetScanCodeSet               = 0xF0,
    ComEcho                         = 0xEE,
    ComSetResetLEDs                 = 0xED
};

struct KeyboardButton {
    int index;
    QString label;
    unsigned char pressed;
    KeyCode makeCode;
    KeyCode breakCode;
    CustomButton* button;
    QGraphicsProxyWidget* proxy;
    int position[2];
    int size[2];
    int fontSize;
};

struct KeyboardMemory {
    QVector<KeyCode *> mBuffer;         // Main Buffer
    KeyCode * tBuffer;                  // TypeMatic Buffer
    KeyCode lastByteSentBuffer;         // Last Byte Sent Buffer
    Chunk outChunk;                     // Outgoing chunk (Answer or Data)
    unsigned short int * oBuffer;       // Output Buffer (Byte)
    unsigned short int iBuffer;         // Host Sent Data Buffer (Command/Argument)
    bool iBufferIsArgument;
    HostCommandSet currentHostCommand;  // Host Command beeing processed
    unsigned char leds;                 // LEDs : NumLock, CapsLock, ScrollLock
};

struct KeyboardInternals {
    uint64_t halfPeriod;
    uint64_t quarterPeriod;
    unsigned char phaseCounter; // 1 trough 44
    Phases phase;               // 0&1: High Quarters, 3&4 Low Quarters
    double typematicDelay;
    double typematicRate;
    double typematicDelaySteps;
    double typematicRateSteps;
    bool isGrabbed;
    QVector<KeyboardButton *> pressedButtons;
    unsigned short int bitPosition;
    int byteIndex;
    uint64_t delayTimeCounter;
    uint64_t rateTimeCounter;
};


class Keyboard : public Component, public eElement
{
    public:
        Keyboard( QString type, QString id );
        ~Keyboard();

        static Component* construct( QString type, QString id );
        static LibraryItem* libraryItem();

        virtual void runEvent() override;
        virtual void voltChanged() override;
        virtual void stamp() override;
        virtual void initialize() override;
        virtual void updateStep() override;
        virtual void remove() override;
        virtual void paint( QPainter* p, const QStyleOptionGraphicsItem* option, QWidget* widget ) override;

        virtual void onbuttonclicked();

    protected:
        void buttonDown(int index);
        void buttonUp(int index);
        void setupButtons();
        void setDefaultValues();
        void setKeyboardState(DeviceStates state);
        void setChunk(ChunkType type, KeyCode * code);
        void clearChunk();
        void setLEDs(unsigned char value);
        void calculateSteps();
        void keyPressEvent( QKeyEvent* event );
        void keyReleaseEvent( QKeyEvent* event );

        int keyToIndex(unsigned int nativeScanCode, unsigned int nativeVirtualKey);
        int m_mappings[223];
        KeyboardButton m_buttonList[KEYBOARD_BUTTON_SIZE];
        KeyCode m_answerCodes[7] = { { 1, {0x00} } ,
                                     { 2, {0xAB, 0x83} },
                                     { 1, {0xAA} } ,
                                     { 1, {0x02} } ,
                                     { 1, {0xEE} } ,
                                     { 1, {0xFA} } ,
                                     { 1, {0xFE} }
        };
        double m_KbdRepeatRate[32] = {
            30.0, 26.7, 24.0, 21.8, 20.7, 18.5, 17.1, 16.0,
            15.0, 13.3, 12.0, 10.9, 10.0,  9.2,  8.6,  8.0,
             7.5,  6.7,  6.0,  5.5,  5.0,  4.6,  4.3,  4.0,
             3.7,  3.3,  3.0,  2.7,  2.5,  2.3,  2.1,  2.0
        };
        double m_KbdDelay[4] = { 0.25, 0.50, 0.75, 1.0 };

        // Open collector communication pins
        IoPin* m_pinClock;
        IoPin* m_pinData;

        KeyboardMemory m_kbdMemory;
        DeviceStates m_kbdState;
        KeyboardInternals m_kbd;

        // Grab keyboard button
        CustomButton* m_button;

        QGraphicsProxyWidget* m_proxy;

};

#endif
