/***************************************************************************
 *   Copyright (C) 2024 by Theodore H. Kaskalis                            *
 *                                                                         *
 ***( see copyright.txt file at root folder )*******************************/

#include "keyboard.h"
#include "simulator.h"
#include "itemlibrary.h"
#include "connector.h"

#define tr(str) simulideTr("Keyboard",str)

Component* Keyboard::construct( QString type, QString id )
{ return new Keyboard( type, id ); }

LibraryItem* Keyboard::libraryItem()
{
    return new LibraryItem(
        tr("Keyboard"),
        "Switches",
        "keyboard.png",
        "Keyboard",
        Keyboard::construct);
}

Keyboard::Keyboard( QString type, QString id )
      : Component( type, id )
      , eElement( id )
{
    m_graphical = true;
    Circuit::self()->accepKeys(true);

    m_pinClock = new IoPin ( 180, QPoint( -236, -44), id+"-PinClock"  , 0, this, openCo );
    m_pinData  = new IoPin ( 180, QPoint( -236, -28), id+"-PinData"   , 0, this, openCo );

    Simulator::self()->addToUpdateList( this );

    m_area = QRectF( -228, -60, 456, 120 );

    m_pinClock->setLabelText( "Clock" );
    m_pinClock->setInputHighV( 2.4 );
    m_pinClock->setInputLowV( 0.7 );
    m_pinClock->setOutLowV( 0 );

    m_pinData->setLabelText( "Data" );
    m_pinData->setInputHighV( 2.4 );
    m_pinData->setInputLowV( 0.7 );
    m_pinData->setOutLowV( 0 );

    m_pin.resize( 2 );
    m_pin[0] = m_pinClock;
    m_pin[1] = m_pinData;

    m_button = new CustomButton( );
    m_button->setFixedSize( 28, 72 );
    m_button->setCheckable( false );
    m_button->setDisabled( true );
    m_button->setStyleSheet("font-size: 8px;");
    m_button->setText("\nGRAB\n\nreal\nkey-\nboard");
    m_kbd.isGrabbed = false;
    m_kbdMemory.leds = 0b000;

    m_kbd.halfPeriod = KEYBOARD_BASE_STEPS/(2.0*KEYBOARD_FREQUENCY);
    m_kbd.quarterPeriod = m_kbd.halfPeriod / 2.0;

    m_proxy = Circuit::self()->addWidget( m_button );
    m_proxy->setParentItem( this );
    m_proxy->setPos( -224, -15);
    m_proxy->setTransformOriginPoint( m_proxy->boundingRect().center() );

    QObject::connect( m_button, &CustomButton::clicked, [=](){ onbuttonclicked(); });

    for (int i = 0; i < 223; i++) {
        m_mappings[i] = KEYBOARD_BUTTON_SIZE;
    }

#ifdef Q_OS_WIN
    m_mappings[ 27] =   0;
    m_mappings[112] =   1;
    m_mappings[113] =   2;
    m_mappings[114] =   3;
    m_mappings[115] =   4;
    m_mappings[116] =   5;
    m_mappings[117] =   6;
    m_mappings[118] =   7;
    m_mappings[119] =   8;
    m_mappings[120] =   9;
    m_mappings[121] =  10;
    m_mappings[122] =  11;
    m_mappings[123] =  12;
    m_mappings[ 44] =  13;
    m_mappings[145] =  14;
    m_mappings[ 19] =  15;
    m_mappings[192] =  16;
    m_mappings[ 49] =  17;
    m_mappings[ 50] =  18;
    m_mappings[ 51] =  19;
    m_mappings[ 52] =  20;
    m_mappings[ 53] =  21;
    m_mappings[ 54] =  22;
    m_mappings[ 55] =  23;
    m_mappings[ 56] =  24;
    m_mappings[ 57] =  25;
    m_mappings[ 48] =  26;
    m_mappings[189] =  27;
    m_mappings[187] =  28;
    m_mappings[  8] =  29;
    m_mappings[ 45] =  30;
    m_mappings[ 36] =  31;
    m_mappings[ 33] =  32;
    m_mappings[144] =  33;
    m_mappings[111] =  34;
    m_mappings[106] =  35;
    m_mappings[109] =  36;
    m_mappings[  9] =  37;
    m_mappings[ 81] =  38;
    m_mappings[ 87] =  39;
    m_mappings[ 69] =  40;
    m_mappings[ 82] =  41;
    m_mappings[ 84] =  42;
    m_mappings[ 89] =  43;
    m_mappings[ 85] =  44;
    m_mappings[ 73] =  45;
    m_mappings[ 79] =  46;
    m_mappings[ 80] =  47;
    m_mappings[219] =  48;
    m_mappings[221] =  49;
    m_mappings[220] =  50;
    m_mappings[ 46] =  51;
    m_mappings[ 35] =  52;
    m_mappings[ 34] =  53;
    m_mappings[103] =  54;
    m_mappings[104] =  55;
    m_mappings[105] =  56;
    m_mappings[107] =  57;
    m_mappings[ 20] =  58;
    m_mappings[ 65] =  59;
    m_mappings[ 83] =  60;
    m_mappings[ 68] =  61;
    m_mappings[ 70] =  62;
    m_mappings[ 71] =  63;
    m_mappings[ 72] =  64;
    m_mappings[ 74] =  65;
    m_mappings[ 75] =  66;
    m_mappings[ 76] =  67;
    m_mappings[186] =  68;
    m_mappings[222] =  69;
    m_mappings[100] =  71;
    m_mappings[101] =  72;
    m_mappings[102] =  73;
    m_mappings[160] =  74;
    m_mappings[ 90] =  75;
    m_mappings[ 88] =  76;
    m_mappings[ 67] =  77;
    m_mappings[ 86] =  78;
    m_mappings[ 66] =  79;
    m_mappings[ 78] =  80;
    m_mappings[ 77] =  81;
    m_mappings[188] =  82;
    m_mappings[190] =  83;
    m_mappings[191] =  84;
    m_mappings[161] =  85;
    m_mappings[ 38] =  86;
    m_mappings[ 97] =  87;
    m_mappings[ 98] =  88;
    m_mappings[ 99] =  89;
    m_mappings[ 13] =  90;
    m_mappings[162] =  91;
    m_mappings[164] =  92;
    m_mappings[ 32] =  93;
    m_mappings[165] =  94;
    m_mappings[163] =  95;
    m_mappings[ 37] =  96;
    m_mappings[ 40] =  97;
    m_mappings[ 39] =  98;
    m_mappings[ 96] =  99;
    m_mappings[110] = 100;
#endif

#ifdef Q_OS_MACOS
    m_mappings[ 53] =   0;
    m_mappings[122] =   1;
    m_mappings[120] =   2;
    m_mappings[ 99] =   3;
    m_mappings[118] =   4;
    m_mappings[ 96] =   5;
    m_mappings[ 97] =   6;
    m_mappings[ 98] =   7;
    m_mappings[100] =   8;
    m_mappings[101] =   9;
    m_mappings[109] =  10;
    m_mappings[103] =  11;
    m_mappings[111] =  12;
    m_mappings[105] =  13;
    m_mappings[107] =  14;
    m_mappings[113] =  15;
    m_mappings[ 50] =  16;
    m_mappings[ 18] =  17;
    m_mappings[ 19] =  18;
    m_mappings[ 20] =  19;
    m_mappings[ 21] =  20;
    m_mappings[ 23] =  21;
    m_mappings[ 22] =  22;
    m_mappings[ 26] =  23;
    m_mappings[ 28] =  24;
    m_mappings[ 25] =  25;
    m_mappings[ 29] =  26;
    m_mappings[ 27] =  27;
    m_mappings[ 24] =  28;
    m_mappings[ 51] =  29;
    m_mappings[114] =  30;
    m_mappings[115] =  31;
    m_mappings[116] =  32;
    m_mappings[ 71] =  33;
    m_mappings[ 75] =  34;
    m_mappings[ 67] =  35;
    m_mappings[ 78] =  36;
    m_mappings[ 48] =  37;
    m_mappings[ 12] =  38;
    m_mappings[ 13] =  39;
    m_mappings[ 14] =  40;
    m_mappings[ 15] =  41;
    m_mappings[ 17] =  42;
    m_mappings[ 16] =  43;
    m_mappings[ 32] =  44;
    m_mappings[ 34] =  45;
    m_mappings[ 31] =  46;
    m_mappings[ 35] =  47;
    m_mappings[ 33] =  48;
    m_mappings[ 30] =  49;
    m_mappings[ 42] =  50;
    m_mappings[117] =  51;
    m_mappings[119] =  52;
    m_mappings[121] =  53;
    m_mappings[ 89] =  54;
    m_mappings[ 91] =  55;
    m_mappings[ 92] =  56;
    m_mappings[ 69] =  57;
    m_mappings[ 57] =  58;
    m_mappings[  0] =  59;
    m_mappings[  1] =  60;
    m_mappings[  2] =  61;
    m_mappings[  3] =  62;
    m_mappings[  5] =  63;
    m_mappings[  4] =  64;
    m_mappings[ 38] =  65;
    m_mappings[ 40] =  66;
    m_mappings[ 37] =  67;
    m_mappings[ 41] =  68;
    m_mappings[ 39] =  69;
    m_mappings[ 36] =  70;
    m_mappings[ 86] =  71;
    m_mappings[ 87] =  72;
    m_mappings[ 88] =  73;
    m_mappings[ 56] =  74;
    m_mappings[  6] =  75;
    m_mappings[  7] =  76;
    m_mappings[  8] =  77;
    m_mappings[  9] =  78;
    m_mappings[ 11] =  79;
    m_mappings[ 45] =  80;
    m_mappings[ 46] =  81;
    m_mappings[ 43] =  82;
    m_mappings[ 47] =  83;
    m_mappings[ 44] =  84;
    m_mappings[ 60] =  85;
    m_mappings[126] =  86;
    m_mappings[ 83] =  87;
    m_mappings[ 84] =  88;
    m_mappings[ 85] =  89;
    m_mappings[ 76] =  90;
    m_mappings[ 59] =  91;
    m_mappings[ 58] =  92;
    m_mappings[ 49] =  93;
    m_mappings[ 61] =  94;
    m_mappings[ 62] =  95;
    m_mappings[123] =  96;
    m_mappings[125] =  97;
    m_mappings[124] =  98;
    m_mappings[ 82] =  99;
    m_mappings[ 65] = 100;
#endif

#ifdef Q_OS_LINUX
    m_mappings[  9] =   0;
    m_mappings[ 67] =   1;
    m_mappings[ 68] =   2;
    m_mappings[ 69] =   3;
    m_mappings[ 70] =   4;
    m_mappings[ 71] =   5;
    m_mappings[ 72] =   6;
    m_mappings[ 73] =   7;
    m_mappings[ 74] =   8;
    m_mappings[ 75] =   9;
    m_mappings[ 76] =  10;
    m_mappings[ 95] =  11;
    m_mappings[ 96] =  12;
    m_mappings[107] =  13;
    m_mappings[ 78] =  14;
    m_mappings[127] =  15;
    m_mappings[ 49] =  16;
    m_mappings[ 10] =  17;
    m_mappings[ 11] =  18;
    m_mappings[ 12] =  19;
    m_mappings[ 13] =  20;
    m_mappings[ 14] =  21;
    m_mappings[ 15] =  22;
    m_mappings[ 16] =  23;
    m_mappings[ 17] =  24;
    m_mappings[ 18] =  25;
    m_mappings[ 19] =  26;
    m_mappings[ 20] =  27;
    m_mappings[ 21] =  28;
    m_mappings[ 22] =  29;
    m_mappings[118] =  30;
    m_mappings[110] =  31;
    m_mappings[112] =  32;
    m_mappings[ 77] =  33;
    m_mappings[106] =  34;
    m_mappings[ 63] =  35;
    m_mappings[ 82] =  36;
    m_mappings[ 23] =  37;
    m_mappings[ 24] =  38;
    m_mappings[ 25] =  39;
    m_mappings[ 26] =  40;
    m_mappings[ 27] =  41;
    m_mappings[ 28] =  42;
    m_mappings[ 29] =  43;
    m_mappings[ 30] =  44;
    m_mappings[ 31] =  45;
    m_mappings[ 32] =  46;
    m_mappings[ 33] =  47;
    m_mappings[ 34] =  48;
    m_mappings[ 35] =  49;
    m_mappings[ 51] =  50;
    m_mappings[119] =  51;
    m_mappings[115] =  52;
    m_mappings[117] =  53;
    m_mappings[ 79] =  54;
    m_mappings[ 80] =  55;
    m_mappings[ 81] =  56;
    m_mappings[ 86] =  57;
    m_mappings[ 66] =  58;
    m_mappings[ 38] =  59;
    m_mappings[ 39] =  60;
    m_mappings[ 40] =  61;
    m_mappings[ 41] =  62;
    m_mappings[ 42] =  63;
    m_mappings[ 43] =  64;
    m_mappings[ 44] =  65;
    m_mappings[ 45] =  66;
    m_mappings[ 46] =  67;
    m_mappings[ 47] =  68;
    m_mappings[ 48] =  69;
    m_mappings[ 36] =  70;
    m_mappings[ 83] =  71;
    m_mappings[ 84] =  72;
    m_mappings[ 85] =  73;
    m_mappings[ 50] =  74;
    m_mappings[ 52] =  75;
    m_mappings[ 53] =  76;
    m_mappings[ 54] =  77;
    m_mappings[ 55] =  78;
    m_mappings[ 56] =  79;
    m_mappings[ 57] =  80;
    m_mappings[ 58] =  81;
    m_mappings[ 59] =  82;
    m_mappings[ 60] =  83;
    m_mappings[ 61] =  84;
    m_mappings[ 62] =  85;
    m_mappings[111] =  86;
    m_mappings[ 87] =  87;
    m_mappings[ 88] =  88;
    m_mappings[ 89] =  89;
    m_mappings[104] =  90;
    m_mappings[ 37] =  91;
    m_mappings[ 64] =  92;
    m_mappings[ 65] =  93;
    m_mappings[108] =  94;
    m_mappings[105] =  95;
    m_mappings[113] =  96;
    m_mappings[116] =  97;
    m_mappings[114] =  98;
    m_mappings[ 90] =  99;
    m_mappings[ 91] = 100;
#endif

    setupButtons();
}
Keyboard::~Keyboard(){}

void Keyboard::stamp() {
    if( !Simulator::self()->isPaused() ) m_changed = true;
    m_pinClock->stamp();
    m_pinClock->changeCallBack( this, true );
    m_pinData->stamp();
    m_pinData->changeCallBack( this, true );
}

void Keyboard::initialize() {
    m_changed = true;
}

void Keyboard::setDefaultValues() {
    m_kbd.typematicDelay = KEYBOARD_DEFAULT_TYPEMATIC_DELAY;
    m_kbd.typematicRate  = KEYBOARD_DEFAULT_TYPEMATIC_RATE;
    calculateSteps();
}

void Keyboard::setLEDs(unsigned char value) {
    m_kbdMemory.leds = value;
    update();
}

void Keyboard::calculateSteps() {
    m_kbd.typematicDelaySteps = m_kbd.typematicDelay * KEYBOARD_BASE_STEPS;
    m_kbd.typematicRateSteps = KEYBOARD_BASE_STEPS / m_kbd.typematicRate;
}

void Keyboard::setChunk(ChunkType type, KeyCode * code) {
    m_kbd.delayTimeCounter = 0;
    m_kbd.rateTimeCounter  = 0;
    m_kbd.byteIndex = 0;
    m_kbdMemory.outChunk.type = type;
    m_kbdMemory.outChunk.code = code;
    m_kbd.bitPosition = 1; // Initiate Byte transfer
    m_kbdMemory.oBuffer = &(m_kbdMemory.outChunk.code->data[m_kbd.byteIndex]);
    setKeyboardState( DeviceSendsData );
}

void Keyboard::clearChunk() {
    m_kbdMemory.outChunk.type = EmptyData;
    m_kbdMemory.outChunk.code = NULL;
    m_kbdMemory.oBuffer = NULL;
    m_kbd.delayTimeCounter = 0;
    m_kbd.rateTimeCounter  = 0;
}

void Keyboard::setKeyboardState(DeviceStates state) {
    KeyboardButton * kb;

    m_kbdState = state;

    switch ( m_kbdState ) {
        case KbdDisabled         :
            if ( m_kbd.isGrabbed ) {
                m_button->click();
            }
            m_button->setCheckable( false );
            m_button->setDisabled( true );
            m_kbdMemory.mBuffer.clear();
            clearChunk();
            m_kbdMemory.lastByteSentBuffer  = { 1, {0x00} };
            m_kbdMemory.tBuffer             = NULL;
            m_kbdMemory.oBuffer             = NULL;
            m_kbdMemory.iBuffer             = 0;
            m_kbdMemory.iBufferIsArgument   = false;
            m_kbdMemory.currentHostCommand  = ComEmpty;
            m_kbd.delayTimeCounter          = 0;
            m_kbd.rateTimeCounter           = 0;
            while ( m_kbd.pressedButtons.size() ) {
                kb = m_kbd.pressedButtons.takeFirst();
                kb->pressed = 0;
                kb->button->setDown(false);
            }
            for( int i=0; i<KEYBOARD_BUTTON_SIZE; i++ ) {
                kb = &(m_buttonList[i]);
                kb->button->setDisabled( true );
            }
            setDefaultValues();
            Simulator::self()->cancelEvents( this );
            return;
        case BATinProcess        :
            setLEDs(0b111);
            Simulator::self()->cancelEvents( this );
            Simulator::self()->addEvent( KEYBOARD_500_msec, this );
            return;
        case HostSendsData       :
            m_kbdMemory.tBuffer     = NULL;
            m_kbdMemory.iBuffer     = 0;
            m_kbd.delayTimeCounter  = 0;
            m_kbd.rateTimeCounter   = 0;
            m_kbd.phaseCounter      = 0;
            m_kbd.bitPosition       = 0; // Initiate Byte transfer
            Simulator::self()->addEvent( KEYBOARD_50_usec, this );
            return;
        case DeviceHalted        :
            m_kbdMemory.oBuffer = NULL;
            clearChunk();
            Simulator::self()->cancelEvents( this );
            return;
        case KbdEnabled          :
            m_button->setDisabled( false );
            m_button->setCheckable( true );
            for( int i=0; i<KEYBOARD_BUTTON_SIZE; i++ ) {
                kb = &(m_buttonList[i]);
                kb->button->setDisabled( false );
            }
            Simulator::self()->addEvent( KEYBOARD_10_usec, this );
            setKeyboardState( KbdIdle );
            return;
        case DeviceSendsData     :
            m_kbd.phaseCounter = 0;
            Simulator::self()->addEvent( KEYBOARD_50_usec, this );
            return;
        case KbdIdle             :
            return;
    }
}

void Keyboard::buttonDown(int index)
{
    KeyboardButton * kb = &(m_buttonList[index]);

    if ( m_kbdState == KbdDisabled ) {
        kb->pressed = 0;
        kb->button->setDown(false);
        return;
    }

    (kb->pressed)++;
    if ( kb->pressed == 2 ) return;
    kb->button->setDown(true);
    m_kbd.pressedButtons.append(kb);
    m_kbdMemory.tBuffer = &(kb->makeCode);
    m_kbdMemory.mBuffer.append( &(kb->makeCode) );
}

void Keyboard::buttonUp(int index)
{
    KeyboardButton * kb = &(m_buttonList[index]);

    if ( m_kbdState == KbdDisabled ) {
        kb->pressed = 0;
        kb->button->setDown(false);
        return;
    }

    (kb->pressed)--;

    if ( kb->pressed == 1 ) {
        kb->button->setDown(true);
        return;
    }

    kb->button->setDown(false);

    if ( m_kbd.pressedButtons.contains(kb) ) {
        if ( m_kbdMemory.tBuffer == &(kb->makeCode) ) m_kbdMemory.tBuffer = NULL;
        m_kbd.pressedButtons.removeOne(kb);
        if ( kb->breakCode.size ) m_kbdMemory.mBuffer.append( &(kb->breakCode) ); // Avoid Null Pause/Break Break Code
    }
}

void Keyboard::runEvent()
{
    switch ( m_kbdState ) {
        case BATinProcess:
            setLEDs(0b000);
            setChunk(Answer, &(m_answerCodes[CodeBATPassed]));
            return;
        case KbdIdle:
            if ( m_kbdMemory.mBuffer.size() ) {
                setChunk(BufferData, m_kbdMemory.mBuffer.first());
                return;
            }
            if ( m_kbdMemory.tBuffer ) {
                if ( m_kbd.delayTimeCounter ) {
                    if ( m_kbd.delayTimeCounter + KEYBOARD_50_usec >= m_kbd.typematicDelaySteps ) {
                        setChunk( TypematicData, m_kbdMemory.tBuffer );
                        return;
                    }
                    m_kbd.delayTimeCounter += KEYBOARD_10_usec;
                }
                else if ( m_kbd.rateTimeCounter ) {
                    if ( m_kbd.rateTimeCounter + KEYBOARD_50_usec >= m_kbd.typematicRateSteps ) {
                        setChunk( TypematicData, m_kbdMemory.tBuffer );
                        return;
                    }
                    m_kbd.rateTimeCounter += KEYBOARD_10_usec;
                }
            }
            Simulator::self()->addEvent( KEYBOARD_10_usec, this );
            return;
        case DeviceSendsData  :
            m_kbd.phase = (Phases) (m_kbd.phaseCounter % 4);
            (m_kbd.phaseCounter)++;
            switch ( m_kbd.phase ) {
                case firstHighQuarter:
                    m_pinClock->setOutState(true);
                    if ( m_kbd.phaseCounter > 44 ) { // Byte transfer completed
                        if ( *(m_kbdMemory.oBuffer) != 0xFE ) {
                            m_kbdMemory.lastByteSentBuffer.data[0] = *(m_kbdMemory.oBuffer);
                        }
                        m_kbd.byteIndex++;
                        if ( m_kbd.byteIndex == m_kbdMemory.outChunk.code->size ) { // Chunk transfer completed
                            switch ( m_kbdMemory.outChunk.type ) {
                                case Answer:
                                    if ( m_kbdMemory.outChunk.code == &(m_answerCodes[CodeAcknowledge]) ) {
                                        // Command or Argument Acknowledged
                                        switch ( m_kbdMemory.currentHostCommand ) { // Process Command
                                            case ComReset:
                                                m_kbdMemory.currentHostCommand = ComEmpty;
                                                m_changed = true;
                                                update();
                                                return;
                                            case ComResend:
                                                m_kbdMemory.currentHostCommand = ComEmpty;
                                                setChunk(BufferData, &(m_kbdMemory.lastByteSentBuffer));
                                                break;
                                            case ComSetDefault:
                                                m_kbdMemory.currentHostCommand = ComEmpty;
                                                setDefaultValues();
                                                break;
                                            case ComDisable:
                                                m_kbdMemory.currentHostCommand = ComEmpty;
                                                setKeyboardState(KbdDisabled);
                                                break;
                                            case ComEnable:
                                                m_kbdMemory.currentHostCommand = ComEmpty;
                                                setKeyboardState(KbdEnabled);
                                                return;
                                            case ComReadID:
                                                m_kbdMemory.currentHostCommand = ComEmpty;
                                                setChunk(Answer, &(m_answerCodes[CodeKeyboardID]));
                                                return;
                                            case ComSetScanCodeSet:
                                            case ComSetResetLEDs:
                                            case ComSetTypematicRateDelay:
                                                if ( !(m_kbdMemory.iBufferIsArgument) ) {
                                                    m_kbdMemory.iBufferIsArgument = true;
                                                    setKeyboardState( DeviceHalted );
                                                    return;
                                                }
                                                // Process Argument
                                                switch ( m_kbdMemory.currentHostCommand ) { // Depending on Command
                                                    case ComSetScanCodeSet:
                                                        switch ( m_kbdMemory.iBuffer ) {
                                                            case 0x00:
                                                                m_kbdMemory.iBufferIsArgument   = false;
                                                                m_kbdMemory.currentHostCommand  = ComEmpty;
                                                                setChunk(Answer, &(m_answerCodes[CodeScanCodeSet2]));
                                                                return;
                                                            case 0x01:
                                                            case 0x03:
                                                                qDebug() << "Only Scan Code Set 2 supported!";
                                                            case 0x02:
                                                                break;
                                                            default:
                                                                qDebug() << "Illegal Set Scan Code Set Argument!";
                                                        }
                                                        break;
                                                    case ComSetResetLEDs:
                                                        setLEDs( (unsigned char) m_kbdMemory.iBuffer );
                                                        break;
                                                    case ComSetTypematicRateDelay:
                                                        m_kbd.typematicRate  = m_KbdRepeatRate[ m_kbdMemory.iBuffer & 0b0011111 ];
                                                        m_kbd.typematicDelay = m_KbdDelay[ ( m_kbdMemory.iBuffer & 0b1100000 ) >> 5 ];
                                                        calculateSteps();
                                                        break;
                                                    default:
                                                        qDebug() << "Incorrectly Processing Argument";
                                                }
                                                m_kbdMemory.currentHostCommand  = ComEmpty;
                                                m_kbdMemory.iBufferIsArgument   = false;
                                                break;
                                            default:
                                                break;
                                        }
                                    }
                                    break;
                                case BufferData:
                                    m_kbdMemory.mBuffer.removeFirst();
                                default:
                                    break;
                            }

                            if ( m_kbdMemory.outChunk.type == Answer && m_kbdMemory.outChunk.code == &(m_answerCodes[CodeBATPassed]) ) {
                                setKeyboardState( KbdEnabled );
                                clearChunk();
                                return;
                            }
                            if ( m_kbdMemory.mBuffer.isEmpty() ) {
                                if ( m_kbdMemory.tBuffer ) {
                                    if ( m_kbdMemory.outChunk.type == TypematicData ) {
                                        clearChunk();
                                        m_kbd.rateTimeCounter = (uint64_t) KEYBOARD_10_usec;
                                    }
                                    else {
                                        clearChunk();
                                        m_kbd.delayTimeCounter = (uint64_t) KEYBOARD_10_usec;
                                    }
                                }
                                else clearChunk();
                                Simulator::self()->addEvent( KEYBOARD_10_usec, this );
                                setKeyboardState( KbdIdle );
                                return;
                            }
                            setChunk(BufferData, m_kbdMemory.mBuffer.first());
                            return;
                        }
                        m_kbdMemory.oBuffer = &(m_kbdMemory.outChunk.code->data[m_kbd.byteIndex]);
                        m_kbd.bitPosition = 1; // Initiate new Byte transfer
                        m_kbd.phaseCounter = 0;
                        Simulator::self()->addEvent( KEYBOARD_50_usec, this );
                        return;
                    }
                    break;
                case secondHighQuarter:
                    m_pinData->setOutState( *(m_kbdMemory.oBuffer) & m_kbd.bitPosition ); // Send Bit
                    m_kbd.bitPosition = m_kbd.bitPosition << 1;
                    break;
                case firstLowQuarter:
                    m_pinClock->setOutState(false);
                    break;
                case secondLowQuarter:
                    break;
            }
            Simulator::self()->addEvent( m_kbd.quarterPeriod, this ); // Start counting quarter period
            return;
        case HostSendsData:
            m_kbd.phase = (Phases) (m_kbd.phaseCounter % 4);
            (m_kbd.phaseCounter)++;
            switch ( m_kbd.phase ) {
                case firstHighQuarter:
                    m_pinClock->setOutState(true);
                    break;
                case secondHighQuarter:
                    if ( m_kbd.phaseCounter == 42 ) {
                        m_pinData->setOutState(false); // ACK start
                    }
                    else if ( m_kbd.phaseCounter == 46 ) {
                        m_pinData->setOutState(true); // ACK end
                        // Process Data
                        m_kbdMemory.iBuffer = m_kbdMemory.iBuffer >> 1;
                        int sumOfOnes = 0;
                        unsigned short int tempByte = m_kbdMemory.iBuffer;
                        unsigned short int tmpParity = tempByte >> 8;
                        unsigned short int tmpLSB = 0;
                        for( int k=0; k<8; k++ ) {
                            tmpLSB = tempByte & 1;
                            sumOfOnes += tmpLSB;
                            tempByte = tempByte >> 1;
                        }
                        if ( tmpParity == ( sumOfOnes % 2 ) ) { // Parity error !!
                            qDebug() << "Parity Error!";
                            setChunk(Answer, &(m_answerCodes[CodeResend]));
                            m_kbdMemory.iBuffer = 0;
                            Simulator::self()->addEvent( KEYBOARD_10_usec, this );
                            setKeyboardState( KbdIdle );
                            return;
                        }
                        m_kbdMemory.iBuffer = m_kbdMemory.iBuffer & 0xFF; // Keep a single byte
                        if ( m_kbdMemory.currentHostCommand == ComSetTypematicRateDelay ||
                             m_kbdMemory.currentHostCommand == ComSetScanCodeSet        ||
                             m_kbdMemory.currentHostCommand == ComSetResetLEDs ) { // Process Argument
                            setChunk(Answer, &(m_answerCodes[CodeAcknowledge]));
                            return;
                        }
                        switch ( m_kbdMemory.iBuffer ) {
                            case 0xFF: // Reset
                                m_kbdMemory.currentHostCommand = ComReset;
                                break;
                            case 0xFE: // Resend
                                m_kbdMemory.currentHostCommand = ComResend;
                                break;
                            case 0xF6: // Set Default
                                m_kbdMemory.currentHostCommand = ComSetDefault;
                                break;
                            case 0xF5: // Disable
                                m_kbdMemory.currentHostCommand = ComDisable;
                                break;
                            case 0xF4: // Enable
                                m_kbdMemory.currentHostCommand = ComEnable;
                                break;
                            case 0xF3: // Set Typematic Rate/Delay
                                m_kbdMemory.currentHostCommand = ComSetTypematicRateDelay;
                                break;
                            case 0xF2: // Read ID
                                m_kbdMemory.currentHostCommand = ComReadID;
                                break;
                            case 0xF0: // Set Scan Code Set
                                m_kbdMemory.currentHostCommand = ComSetScanCodeSet;
                                break;
                            case 0xEE: // Echo
                                m_kbdMemory.currentHostCommand = ComEcho;
                                break;
                            case 0xED: // Set/Reset LEDs
                                m_kbdMemory.currentHostCommand = ComSetResetLEDs;
                                break;
                            default:
                                // Unknown (or unsupported) command
                                m_kbdMemory.currentHostCommand = ComEmpty;
                                qDebug() << "Unsupported command";
                                return;
                        }
                        if ( m_kbdMemory.currentHostCommand == ComEcho ) {
                            m_kbdMemory.currentHostCommand = ComEmpty;
                            setChunk(Answer, &(m_answerCodes[CodeEcho]));
                            return;
                        }
                        setChunk(Answer, &(m_answerCodes[CodeAcknowledge]));
                        return;
                    }
                    else {
                        m_kbdMemory.iBuffer |= ((unsigned short int) m_pinData->getInpState()) << m_kbd.bitPosition;
                        (m_kbd.bitPosition)++;
                    }
                    break;
                case firstLowQuarter:
                    m_pinClock->setOutState(false);
                    break;
                case secondLowQuarter:
                    break;
            }
            Simulator::self()->addEvent( m_kbd.quarterPeriod, this ); // Start counting quarter period
        default:
            return;
    }
}

void Keyboard::voltChanged() {
    if ( m_kbdState == BATinProcess ) return;

    if ( m_kbdState == DeviceHalted ) {
        if ( m_pinClock->getInpState() ) { // Host released the clock
            if ( !(m_pinData->getInpState()) ) { // Host wants to send command
                setKeyboardState( HostSendsData );
                return;
            }
            // Resume proper operation
            Simulator::self()->addEvent( KEYBOARD_10_usec, this );
            setKeyboardState( KbdIdle );
        }
        return;
    }

    // Device does NOT drive the clock and Host drives the clock LOW
    if ( m_pinClock->getOutState() && !(m_pinClock->getInpState()) ) setKeyboardState( DeviceHalted );
}

void Keyboard::setupButtons()
{
    int x = -193;
    int y = -57;

    m_buttonList[  0] = {   0, "Esc",       0, { 1, {0x76} }                  , { 2, {0xF0, 0x76} }                        , new CustomButton(), NULL, {    x, y}, {18, 18}, 6 };
    m_buttonList[  1] = {   1,  "F1",       0, { 1, {0x05} }                  , { 2, {0xF0, 0x05} }                        , new CustomButton(), NULL, {x+=32, y}, {18, 18}, 7 };
    m_buttonList[  2] = {   2,  "F2",       0, { 1, {0x06} }                  , { 2, {0xF0, 0x06} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 7 };
    m_buttonList[  3] = {   3,  "F3",       0, { 1, {0x04} }                  , { 2, {0xF0, 0x04} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 7 };
    m_buttonList[  4] = {   4,  "F4",       0, { 1, {0x0C} }                  , { 2, {0xF0, 0x0C} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 7 };
    m_buttonList[  5] = {   5,  "F5",       0, { 1, {0x03} }                  , { 2, {0xF0, 0x03} }                        , new CustomButton(), NULL, {x+=29, y}, {18, 18}, 7 };
    m_buttonList[  6] = {   6,  "F6",       0, { 1, {0x0B} }                  , { 2, {0xF0, 0x0B} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 7 };
    m_buttonList[  7] = {   7,  "F7",       0, { 1, {0x83} }                  , { 2, {0xF0, 0x83} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 7 };
    m_buttonList[  8] = {   8,  "F8",       0, { 1, {0x0A} }                  , { 2, {0xF0, 0x0A} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 7 };
    m_buttonList[  9] = {   9,  "F9",       0, { 1, {0x01} }                  , { 2, {0xF0, 0x01} }                        , new CustomButton(), NULL, {x+=29, y}, {18, 18}, 7 };
    m_buttonList[ 10] = {  10, "F10",       0, { 1, {0x09} }                  , { 2, {0xF0, 0x09} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 7 };
    m_buttonList[ 11] = {  11, "F11",       0, { 1, {0x78} }                  , { 2, {0xF0, 0x78} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 7 };
    m_buttonList[ 12] = {  12, "F12",       0, { 1, {0x07} }                  , { 2, {0xF0, 0x07} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 7 };
    m_buttonList[ 13] = {  13, "Prt\nScr",  0, { 4, {0xE0, 0x12, 0xE0, 0x7C} }, { 6, {0xE0, 0xF0, 0x7C, 0xE0, 0xF0, 0x12} }, new CustomButton(), NULL, {x+=29, y}, {18, 18}, 6 };
    m_buttonList[ 14] = {  14, "Scr\nLock", 0, { 1, {0x7E} }                  , { 2, {0xF0, 0x7E} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 15] = {  15, "Paus\nBrk", 0, { 8, {0xE1, 0x14, 0x77, 0xE1, 0xF0, 0x14, 0xF0, 0x77} }, { 0, {0x00} }      , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };

    x = -193;
    y += 24;
    m_buttonList[ 16] = {  16, "~\n`",      0, { 1, {0x0E} }                  , { 2, {0xF0, 0x0E} }                        , new CustomButton(), NULL, {    x, y}, {18, 18}, 6 };
    m_buttonList[ 17] = {  17, "!\n1",      0, { 1, {0x16} }                  , { 2, {0xF0, 0x16} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 18] = {  18, "@\n2",      0, { 1, {0x1E} }                  , { 2, {0xF0, 0x1E} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 19] = {  19, "#\n3",      0, { 1, {0x26} }                  , { 2, {0xF0, 0x26} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 20] = {  20, "$\n4",      0, { 1, {0x25} }                  , { 2, {0xF0, 0x25} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 21] = {  21, "%\n5",      0, { 1, {0x2E} }                  , { 2, {0xF0, 0x2E} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 22] = {  22, "^\n6",      0, { 1, {0x36} }                  , { 2, {0xF0, 0x36} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 23] = {  23, "&\n7",      0, { 1, {0x3D} }                  , { 2, {0xF0, 0x3D} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 24] = {  24, "*\n8",      0, { 1, {0x3E} }                  , { 2, {0xF0, 0x3E} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 25] = {  25, "(\n9",      0, { 1, {0x46} }                  , { 2, {0xF0, 0x46} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 26] = {  26, ")\n0",      0, { 1, {0x45} }                  , { 2, {0xF0, 0x45} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 27] = {  27, "_\n-",      0, { 1, {0x4E} }                  , { 2, {0xF0, 0x4E} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 28] = {  28, "+\n=",      0, { 1, {0x55} }                  , { 2, {0xF0, 0x55} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 29] = {  29, "Backspace", 0, { 1, {0x66} }                  , { 2, {0xF0, 0x66} }                        , new CustomButton(), NULL, {x+=18, y}, {36, 18}, 6 };
    m_buttonList[ 30] = {  30, "Ins",       0, { 2, {0xE0, 0x70} }            , { 3, {0xE0, 0xF0, 0x70} }                  , new CustomButton(), NULL, {x+=47, y}, {18, 18}, 6 };
    m_buttonList[ 31] = {  31, "Hom",       0, { 2, {0xE0, 0x6C} }            , { 3, {0xE0, 0xF0, 0x6C} }                  , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 32] = {  32, "Page\nUp",  0, { 2, {0xE0, 0x7D} }            , { 3, {0xE0, 0xF0, 0x7D} }                  , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 33] = {  33, "Num\nLock", 0, { 1, {0x77} }                  , { 2, {0xF0, 0x77} }                        , new CustomButton(), NULL, {x+=29, y}, {18, 18}, 6 };
    m_buttonList[ 34] = {  34, "/",         0, { 2, {0xE0, 0x4A} }            , { 3, {0xE0, 0xF0, 0x4A} }                  , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 8 };
    m_buttonList[ 35] = {  35, "*",         0, { 1, {0x7C} }                  , { 2, {0xF0, 0x7C} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18},10 };
    m_buttonList[ 36] = {  36, "-",         0, { 1, {0x7B} }                  , { 2, {0xF0, 0x7B} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18},10 };

    x = -193;
    y += 18;
    m_buttonList[ 37] = {  37, "Tab",       0, { 1, {0x0D} }                  , { 2, {0xF0, 0x0D} }                        , new CustomButton(), NULL, {    x, y}, {26, 18}, 6 };
    m_buttonList[ 38] = {  38, "Q",         0, { 1, {0x15} }                  , { 2, {0xF0, 0x15} }                        , new CustomButton(), NULL, {x+=26, y}, {18, 18}, 8 };
    m_buttonList[ 39] = {  39, "W",         0, { 1, {0x1D} }                  , { 2, {0xF0, 0x1D} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 8 };
    m_buttonList[ 40] = {  40, "E",         0, { 1, {0x24} }                  , { 2, {0xF0, 0x24} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 8 };
    m_buttonList[ 41] = {  41, "R",         0, { 1, {0x2D} }                  , { 2, {0xF0, 0x2D} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 8 };
    m_buttonList[ 42] = {  42, "T",         0, { 1, {0x2C} }                  , { 2, {0xF0, 0x2C} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 8 };
    m_buttonList[ 43] = {  43, "Y",         0, { 1, {0x35} }                  , { 2, {0xF0, 0x35} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 8 };
    m_buttonList[ 44] = {  44, "U",         0, { 1, {0x3C} }                  , { 2, {0xF0, 0x3C} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 8 };
    m_buttonList[ 45] = {  45, "I",         0, { 1, {0x43} }                  , { 2, {0xF0, 0x43} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 8 };
    m_buttonList[ 46] = {  46, "O",         0, { 1, {0x44} }                  , { 2, {0xF0, 0x44} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 8 };
    m_buttonList[ 47] = {  47, "P",         0, { 1, {0x4D} }                  , { 2, {0xF0, 0x4D} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 8 };
    m_buttonList[ 48] = {  48, "{\n[",      0, { 1, {0x54} }                  , { 2, {0xF0, 0x54} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 49] = {  49, "}\n]",      0, { 1, {0x5B} }                  , { 2, {0xF0, 0x5B} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 50] = {  50, "|\n\\",     0, { 1, {0x5D} }                  , { 2, {0xF0, 0x5D} }                        , new CustomButton(), NULL, {x+=18, y}, {28, 18}, 6 };
    m_buttonList[ 51] = {  51, "Del",       0, { 2, {0xE0, 0x71} }            , { 3, {0xE0, 0xF0, 0x71} }                  , new CustomButton(), NULL, {x+=39, y}, {18, 18}, 6 };
    m_buttonList[ 52] = {  52, "End",       0, { 2, {0xE0, 0x69} }            , { 3, {0xE0, 0xF0, 0x69} }                  , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 53] = {  53, "Page\nDn",  0, { 2, {0xE0, 0x7A} }            , { 3, {0xE0, 0xF0, 0x7A} }                  , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 54] = {  54, "7\nHom",    0, { 1, {0x6C} }                  , { 2, {0xF0, 0x6C} }                        , new CustomButton(), NULL, {x+=29, y}, {18, 18}, 6 };
    m_buttonList[ 55] = {  55, "8\n\u25B2", 0, { 1, {0x75} }                  , { 2, {0xF0, 0x75} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 56] = {  56, "9\nPgUp",   0, { 1, {0x7D} }                  , { 2, {0xF0, 0x7D} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 57] = {  57, "+",         0, { 1, {0x79} }                  , { 2, {0xF0, 0x79} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 36},10 };

    x = -193;
    y += 18;
    m_buttonList[ 58] = {  58, "Caps\nLock",0, { 1, {0x58} }                  , { 2, {0xF0, 0x58} }                        , new CustomButton(), NULL, {    x, y}, {32, 18}, 6 };
    m_buttonList[ 59] = {  59, "A",         0, { 1, {0x1C} }                  , { 2, {0xF0, 0x1C} }                        , new CustomButton(), NULL, {x+=32, y}, {18, 18}, 8 };
    m_buttonList[ 60] = {  60, "S",         0, { 1, {0x1B} }                  , { 2, {0xF0, 0x1B} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 8 };
    m_buttonList[ 61] = {  61, "D",         0, { 1, {0x23} }                  , { 2, {0xF0, 0x23} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 8 };
    m_buttonList[ 62] = {  62, "F",         0, { 1, {0x2B} }                  , { 2, {0xF0, 0x2B} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 8 };
    m_buttonList[ 63] = {  63, "G",         0, { 1, {0x34} }                  , { 2, {0xF0, 0x34} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 8 };
    m_buttonList[ 64] = {  64, "H",         0, { 1, {0x33} }                  , { 2, {0xF0, 0x33} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 8 };
    m_buttonList[ 65] = {  65, "J",         0, { 1, {0x3B} }                  , { 2, {0xF0, 0x3B} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 8 };
    m_buttonList[ 66] = {  66, "K",         0, { 1, {0x42} }                  , { 2, {0xF0, 0x42} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 8 };
    m_buttonList[ 67] = {  67, "L",         0, { 1, {0x4B} }                  , { 2, {0xF0, 0x4B} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 8 };
    m_buttonList[ 68] = {  68, ":\n;",      0, { 1, {0x4C} }                  , { 2, {0xF0, 0x4C} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 69] = {  69, "\"\n'",     0, { 1, {0x52} }                  , { 2, {0xF0, 0x52} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 70] = {  70, "Enter",     0, { 1, {0x5A} }                  , { 2, {0xF0, 0x5A} }                        , new CustomButton(), NULL, {x+=18, y}, {40, 18}, 6 };
    m_buttonList[ 71] = {  71, "4\n\u25C0", 0, { 1, {0x6B} }                  , { 2, {0xF0, 0x6B} }                        , new CustomButton(), NULL, {x+=116,y}, {18, 18}, 6 };
    m_buttonList[ 72] = {  72, "5\n",       0, { 1, {0x73} }                  , { 2, {0xF0, 0x73} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 73] = {  73, "6\n\u25B6", 0, { 1, {0x74} }                  , { 2, {0xF0, 0x74} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };

    x = -193;
    y += 18;
    m_buttonList[ 74] = {  74, "Shift",     0, { 1, {0x12} }                  , { 2, {0xF0, 0x12} }                        , new CustomButton(), NULL, {    x, y}, {43, 18}, 6 };
    m_buttonList[ 75] = {  75, "Z",         0, { 1, {0x1A} }                  , { 2, {0xF0, 0x1A} }                        , new CustomButton(), NULL, {x+=43, y}, {18, 18}, 8 };
    m_buttonList[ 76] = {  76, "X",         0, { 1, {0x22} }                  , { 2, {0xF0, 0x22} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 8 };
    m_buttonList[ 77] = {  77, "C",         0, { 1, {0x21} }                  , { 2, {0xF0, 0x21} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 8 };
    m_buttonList[ 78] = {  78, "V",         0, { 1, {0x2A} }                  , { 2, {0xF0, 0x2A} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 8 };
    m_buttonList[ 79] = {  79, "B",         0, { 1, {0x32} }                  , { 2, {0xF0, 0x32} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 8 };
    m_buttonList[ 80] = {  80, "N",         0, { 1, {0x31} }                  , { 2, {0xF0, 0x31} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 8 };
    m_buttonList[ 81] = {  81, "M",         0, { 1, {0x3A} }                  , { 2, {0xF0, 0x3A} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 8 };
    m_buttonList[ 82] = {  82, "<\n,",      0, { 1, {0x41} }                  , { 2, {0xF0, 0x41} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 83] = {  83, ">\n.",      0, { 1, {0x49} }                  , { 2, {0xF0, 0x49} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 84] = {  84, "?\n/",      0, { 1, {0x4A} }                  , { 2, {0xF0, 0x4A} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 85] = {  85, "Shift",     0, { 1, {0x59} }                  , { 2, {0xF0, 0x59} }                        , new CustomButton(), NULL, {x+=18, y}, {47, 18}, 6 };
    m_buttonList[ 86] = {  86, "\u25B2",    0, { 2, {0xE0, 0x75} }            , { 3, {0xE0, 0xF0, 0x75} }                  , new CustomButton(), NULL, {x+=76, y}, {18, 18}, 6 };
    m_buttonList[ 87] = {  87, "1\nEnd",    0, { 1, {0x69} }                  , { 2, {0xF0, 0x69} }                        , new CustomButton(), NULL, {x+=47, y}, {18, 18}, 6 };
    m_buttonList[ 88] = {  88, "2\n\u25BC", 0, { 1, {0x72} }                  , { 2, {0xF0, 0x72} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 89] = {  89, "3\nPgDn",   0, { 1, {0x7A} }                  , { 2, {0xF0, 0x7A} }                        , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 90] = {  90, "Ent",       0, { 2, {0xE0, 0x5A} }            , { 3, {0xE0, 0xF0, 0x5A} }                  , new CustomButton(), NULL, {x+=18, y}, {18, 36}, 6 };

    x = -193;
    y += 18;
    m_buttonList[ 91] = {  91, "Ctrl",      0, { 1, {0x14} }                  , { 2, {0xF0, 0x14} }                        , new CustomButton(), NULL, {    x, y}, {26, 18}, 6 };
    m_buttonList[ 92] = {  92, "Alt",       0, { 1, {0x11} }                  , { 2, {0xF0, 0x11} }                        , new CustomButton(), NULL, {x+=47, y}, {26, 18}, 6 };
    m_buttonList[ 93] = {  93, "",          0, { 1, {0x29} }                  , { 2, {0xF0, 0x29} }                        , new CustomButton(), NULL, {x+=26, y},{125, 18}, 6 };
    m_buttonList[ 94] = {  94, "Alt",       0, { 2, {0xE0, 0x11} }            , { 3, {0xE0, 0xF0, 0x11} }                  , new CustomButton(), NULL, {x+=125,y}, {26, 18}, 6 };
    m_buttonList[ 95] = {  95, "Ctrl",      0, { 2, {0xE0, 0x14} }            , { 3, {0xE0, 0xF0, 0x14} }                  , new CustomButton(), NULL, {x+=46, y}, {26, 18}, 6 };
    m_buttonList[ 96] = {  96, "\u25C0",    0, { 2, {0xE0, 0x6B} }            , { 3, {0xE0, 0xF0, 0x6B} }                  , new CustomButton(), NULL, {x+=37, y}, {18, 18}, 6 };
    m_buttonList[ 97] = {  97, "\u25BC",    0, { 2, {0xE0, 0x72} }            , { 3, {0xE0, 0xF0, 0x72} }                  , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 98] = {  98, "\u25B6",    0, { 2, {0xE0, 0x74} }            , { 3, {0xE0, 0xF0, 0x74} }                  , new CustomButton(), NULL, {x+=18, y}, {18, 18}, 6 };
    m_buttonList[ 99] = {  99, "0\nIns",    0, { 1, {0x70} }                  , { 2, {0xF0, 0x70} }                        , new CustomButton(), NULL, {x+=29, y}, {36, 18}, 6 };
    m_buttonList[100] = { 100, "Del\n.",    0, { 1, {0x71} }                  , { 2, {0xF0, 0x71} }                        , new CustomButton(), NULL, {x+=36, y}, {18, 18}, 6 };

    KeyboardButton * kb;
    int sumOfOnes;
    unsigned short int tempByte;

    for( int i=0; i<KEYBOARD_BUTTON_SIZE; i++ )
    {
        int j;

        // Append Parity, Start and Stop bits
        kb = &(m_buttonList[i]);
        for( j=0; j<kb->makeCode.size; j++ ) {
            tempByte = kb->makeCode.data[j];
            sumOfOnes = 0;
            for( int k=0; k<8; k++ ) {
                sumOfOnes += tempByte & 1;
                tempByte = tempByte >> 1;
            }
            tempByte = ( sumOfOnes % 2 ) ? 0b1000000000 : 0b1100000000;
            kb->makeCode.data[j] = ( kb->makeCode.data[j] | tempByte ) << 1;
        }
        for( j=0; j<kb->breakCode.size; j++ ) {
            tempByte = kb->breakCode.data[j];
            sumOfOnes = 0;
            for( int k=0; k<8; k++ ) {
                sumOfOnes += tempByte & 1;
                tempByte = tempByte >> 1;
            }
            tempByte = ( sumOfOnes % 2 ) ? 0b1000000000 : 0b1100000000;
            kb->breakCode.data[j] = ( kb->breakCode.data[j] | tempByte ) << 1;
        }

        kb->button->setStyleSheet(QString("font-size: %1px;").arg(kb->fontSize));
        kb->button->setText(kb->label);
        kb->button->setFixedSize( kb->size[0], kb->size[1] );
        kb->button->setCheckable( false );
        kb->button->setDisabled( true );
        kb->proxy = Circuit::self()->addWidget( kb->button );
        kb->proxy->setParentItem( this );
        kb->proxy->setPos( kb->position[0], kb->position[1]);
        kb->proxy->setTransformOriginPoint( kb->proxy->boundingRect().center() );
        QObject::connect( kb->button, &CustomButton::pressed, [=](){ buttonDown(i); });
        QObject::connect( kb->button, &CustomButton::released, [=](){  buttonUp(i); });
    }

    // Append Parity, Start, Stop and final Byte bits
    for( int i=0; i<7; i++ ) {
        int j;
        for( j=0; j<m_answerCodes[i].size; j++ ) {
            tempByte = m_answerCodes[i].data[j];
            sumOfOnes = 0;
            for( int k=0; k<8; k++ ) {
                sumOfOnes += tempByte & 1;
                tempByte = tempByte >> 1;
            }
            tempByte = ( sumOfOnes % 2 ) ? 0b1000000000 : 0b1100000000;
            m_answerCodes[i].data[j] = ( m_answerCodes[i].data[j] | tempByte ) << 1;
        }
    }
}

#ifdef Q_OS_WIN
int Keyboard::keyToIndex(unsigned int nativeScanCode, unsigned int nativeVirtualKey)
{
    if ( nativeVirtualKey > 222) return KEYBOARD_BUTTON_SIZE;
    if ( nativeVirtualKey == 13 && nativeScanCode == 28 ) return 70;
    return m_mappings[nativeVirtualKey];
}
#endif

#ifdef Q_OS_MACOS
int Keyboard::keyToIndex(unsigned int nativeScanCode, unsigned int nativeVirtualKey)
{
    if ( nativeVirtualKey > 222) return KEYBOARD_BUTTON_SIZE;
    return m_mappings[nativeVirtualKey];
}
#endif

#ifdef Q_OS_LINUX
int Keyboard::keyToIndex(unsigned int nativeScanCode, unsigned int nativeVirtualKey)
{
    if ( nativeScanCode > 222) return KEYBOARD_BUTTON_SIZE;
    return m_mappings[nativeScanCode];
}
#endif

void Keyboard::updateStep()
{
    if( m_changed ) {
        m_changed = false;
        Simulator::self()->cancelEvents( this );
        m_pinClock->setOutState( true );
        m_pinData->setOutState( true );
        setLEDs(0b000);
        setKeyboardState( KbdDisabled );
        if ( Simulator::self()->isRunning() ) {
            setKeyboardState( BATinProcess );
            // After BAT completion code is sent, keyboard is enabled and becomes Idle.
        }
    }
}

void Keyboard::onbuttonclicked()
{
    // while ( m_pressedButtons.size() ) {
    //     buttonUp(m_pressedButtons.last()->index);
    // }

    m_kbd.isGrabbed = m_button->isChecked();
    if( m_kbd.isGrabbed ) {
        m_button->setText("UN\nGRAB\n\nreal\nkey-\nboard");
        Circuit::self()->accepKeys(false);
        grabKeyboard();
    }
    else {
        m_button->setText("\nGRAB\n\nreal\nkey-\nboard");
        Circuit::self()->accepKeys(true);
        ungrabKeyboard();
    }
    update();
}

void Keyboard::remove()
{
    KeyboardButton * kb;

    for( int i=0; i<KEYBOARD_BUTTON_SIZE; i++ ) {
        kb = &(m_buttonList[i]);
        Circuit::self()->removeItem( kb->proxy );
        delete kb->button;
    }

    Component::remove();
}

void Keyboard::keyPressEvent( QKeyEvent* event )
{
    if( m_kbd.isGrabbed && !(event->isAutoRepeat()) )
    {
        int index = keyToIndex( event->nativeScanCode(), event->nativeVirtualKey() );
        if ( index < KEYBOARD_BUTTON_SIZE ) buttonDown(index);
    }
}

void Keyboard::keyReleaseEvent( QKeyEvent* event )
{
    if( m_kbd.isGrabbed && !(event->isAutoRepeat()) )
    {
        int index = keyToIndex( event->nativeScanCode(), event->nativeVirtualKey() );
        if ( index < KEYBOARD_BUTTON_SIZE ) buttonUp(index);
    }
}

void Keyboard::paint( QPainter* p, const QStyleOptionGraphicsItem* option, QWidget* widget )
{
    QFont font = p->font() ;
    font.setPointSize(3);
    p->setFont(font);

    p->setPen( QPen( Qt::black, 1, Qt::SolidLine, Qt::RoundCap, Qt::RoundJoin ) );
    p->setBrush( QColor(50, 70, 100) );
    p->drawRoundedRect( m_area,2,2 );

    p->setBrush( Qt::lightGray );
    p->drawRoundedRect( QRectF( 154, -56, 70, 17 ), 2, 2 );
    p->drawLine( QLineF( QLine( 177, -56, 177, -39) ) );
    p->drawLine( QLineF( QLine( 201, -56, 201, -39) ) );
    p->drawText( QRectF( 153, -57, 25, 14 ), Qt::AlignCenter, "Num\nLock" );
    p->drawText( QRectF( 177, -57, 25, 14 ), Qt::AlignCenter, "Caps\nLock" );
    p->drawText( QRectF( 200, -57, 25, 14 ), Qt::AlignCenter, "Scroll\nLock" );

    p->setPen( QPen( Qt::black, 0.5, Qt::SolidLine, Qt::RoundCap, Qt::RoundJoin ) );
    p->setBrush ( (m_kbdMemory.leds & 0b010) ? Qt::yellow : Qt::black );
    p->drawRoundedRect( QRectF( 161.5, -44, 8, 3 ), 1, 1 );
    p->setBrush ( (m_kbdMemory.leds & 0b100) ? Qt::yellow : Qt::black );
    p->drawRoundedRect( QRectF( 186, -44, 8, 3 ), 1, 1 );
    p->setBrush ( (m_kbdMemory.leds & 0b001) ? Qt::yellow : Qt::black );
    p->drawRoundedRect( QRectF( 208.5, -44, 8, 3 ), 1, 1 );
}
