/***************************************************************************
 *   Copyright (C) 2018 by santiago González                               *
 *   santigoro@gmail.com                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, see <http://www.gnu.org/licenses/>.  *
 *                                                                         *
 ***************************************************************************/


#include <QPainter>
#include <QGraphicsProxyWidget>
#include <cmath>

#include "rotswitch.h"
#include "circuit.h"
#include "itemlibrary.h"
#include "simulator.h"
#include "circuitwidget.h"
#include "iopin.h"
#include "propdialog.h"
#include "component.h"
#include "dialwidget.h"
#include "customdial.h"

#include "intprop.h"
#include "doubleprop.h"

#define tr(str) simulideTr("RotSwitch",str)

Component* RotSwitch::construct( QString type, QString id )
{ return new RotSwitch( type, id ); }

LibraryItem* RotSwitch::libraryItem()
{
    return new LibraryItem(
            tr( "Rotary Switch" ),
            "Switches" ,
            "rotswitch.png",
            "RotSwitch",
            RotSwitch::construct);
}

RotSwitch::RotSwitch( QString type, QString id )
      : Component( type, id )
      , eElement( id )
{
    m_rows = 8;
    m_ways = 1;
    m_graphical = true;
    setLabelPos(-32,m_rows*4, 0);

    m_dial = new CustomDial();
    m_dial->setFixedSize( 24, 24 );
    m_dial->setWrapping( false );
    m_dial->setSingleStep( 1 );
//    m_dial->setNotchTarget( 10 );

    m_proxy = Circuit::self()->addWidget( m_dial );
    m_proxy->setParentItem( this );
    updateProxy();

    Simulator::self()->addToUpdateList( this );

    QObject::connect( m_dial, &QAbstractSlider::valueChanged, [=](int v){ dialChanged(v); } );

    setupButtons();

    addPropGroup( { tr("Main"), {
new DoubProp<RotSwitch>("n_cont", tr("N° cont"),"_3-8", this, &RotSwitch::rows,   &RotSwitch::setRows ),
new IntProp<RotSwitch>("ways", tr("Ways"),"_1-2", this, &RotSwitch::ways,   &RotSwitch::setways )
    }, 0} );
}
RotSwitch::~RotSwitch(){}


void RotSwitch::stamp()
{
    eNode* comNode = m_pin[0]->getEnode();

    for( int row=1; row<m_rows+1; row++ ) {

        eNode* colNode = m_pin[row]->getEnode();

        ePin* epin0 = m_positions.at( row-1 )->getEpin( 0 );
        epin0->setEnode( comNode );

        ePin* epin1 = m_positions.at( row-1 )->getEpin( 1 );
        epin1->setEnode( colNode );
    }

    if (m_ways == 2) {        
       eNode* com1Node = m_pin[m_rows+1]->getEnode();

       for( int row=m_rows+2; row<(m_rows*2)+2; row++ ) {

           eNode* colNode = m_pin[row]->getEnode();

           ePin* epin0 = m_positions.at( row-2 )->getEpin( 0 );
           epin0->setEnode( com1Node );

           ePin* epin1 = m_positions.at( row-2 )->getEpin( 1 );
           epin1->setEnode( colNode );
       }
    }

    m_needUpdate = true;
    updateStep();
}

void RotSwitch::dialChanged( int ) // Called when dial is rotated
{
    m_needUpdate = true;
    if( !Simulator::self()->isRunning() ) updateStep();
}

void RotSwitch::setupButtons()
{
    m_area = QRectF(-28,-(m_rows*4), 56, m_rows*8 );

    if( Simulator::self()->isRunning() )  CircuitWidget::self()->powerCircOff();
      
    for( MechContact* button : m_positions ) {       // Elimina i contatti se ci sono
        m_positions.removeOne( button );
        delete button;
    }
    
    for( Pin* pin : m_pin ) {                        // Elimina i pin se ci sono
         deletePin( pin );
    }

    if (m_ways == 1) {                               // Aggiorna il numero di pin
        m_pin.resize( m_rows + 1 );
    }
    else {
        m_pin.resize( (2*m_rows) + 2 );
    }

    point_pin1.clear();

    for( int row=0; row<m_rows+1; row++ )              // Crea contatti e pin prima via
    {
        QString pinId = m_id;
        pinId.append( QString("-Pin")+QString::number(row)) ;

        if( row == 0 ) {
            QPoint pinPos = QPoint( -35, 0);
            m_pin[row] = new Pin( 180, pinPos, pinId, 0, this);
            m_pin[row]->setLength(4);
//            m_pin[row]->setFontSize(3);
//            m_pin[row]->setLabelText("Com");
        }
        else if (row <= 8 && row > 0) {
            QPoint pinPos = QPoint(35, -((m_rows-1)*4) +((row-1)*8));  // Pin
            point_pin1.append(pinPos);                                 // way 1 Pin position array
            m_pin[row] = new Pin( 0, pinPos, pinId, 0, this);
            m_pin[row]->setLength(4);
//            m_pin[row]->setFontSize(3);
//            m_pin[row]->setLabelText(QString::number(row));
        }

        if( row != 0 ) {
            QString butId = m_id+"button"+QString::number(row);
            MechContact* button = new MechContact( "MechContact", butId );
            button->SetupButton();
            button->setParentItem( this );
            button->setPos( QPointF(25, 16+row*8 ) );
            button->setFlag( QGraphicsItem::ItemIsSelectable, false );
            button->setSwitch(false);
            m_positions.append( button );
        }
    }

    if (m_ways == 2) {                                        // Crea contatti e pin seconda via
        point_pin2.clear();
        m_area = QRectF(-28,-(((m_rows-1)*8+8)+((m_rows-1)*4+8)), 56, (m_rows-1)*16+16 );

        for( int row=m_rows+1; row<(m_rows*2)+2; row++ ) {

           QString pinId = m_id;
           pinId.append( QString("-Pin")+QString::number(row));

           if( row == m_rows +1) {
               QPoint pinPos = QPoint( -35, -(((m_rows-1)*8) +16));
               m_pin[row] = new Pin( 180, pinPos, pinId, 0, this);
               m_pin[row]->setLength(4);
//               m_pin[row]->setFontSize(3);
//               m_pin[row]->setLabelText("Com");
           }
           else if( row > m_rows +1)  {
               QPoint pinPos = QPoint(35, -((m_rows-1)*4)+((row-1)*8)-(((m_rows-1)*8) +16)*2);
               point_pin2.append(pinPos);                                 // way 2 Pin position array
               m_pin[row] = new Pin( 0, pinPos, pinId, 0, this);
               m_pin[row]->setLength(4);
//               m_pin[row]->setFontSize(3);
//               m_pin[row]->setLabelText(QString::number(row-m_rows-1));
           }

           if( row != m_rows +1) {
               QString butId = m_id+"button"+QString::number(row);
               MechContact* button = new MechContact( "MechContact", butId );
               button->SetupButton();
               button->setParentItem( this );
               button->setPos( QPointF(25, 16+row*8 ) );
               button->setFlag( QGraphicsItem::ItemIsSelectable, false );
               button->setSwitch(false);
               m_positions.append( button );
           }
       }
        center1 = QPointF( -12, -(((m_rows-1)*8) +16));
        point_arc1.clear();

        for( int p=0; p < point_pin2.length(); p++ ) {
            QPointF tmp = point_pin2.at(p);
            point_arc1.append(QPointF(sqrt(abs(pow(r,2)-pow(tmp.y()-center1.y(),2))) + center1.x(), tmp.y()));
        }
    }

    m_dial->setMaximum(m_rows-1);
    m_dial->setValue(0);


    center = QPointF( -12, 0);
    point_arc.clear();
    for( int p=0; p < point_pin1.length(); p++ ) {
        QPointF tmp = point_pin1.at(p);
        point_arc.append(QPointF(sqrt(abs(pow(r,2)-pow(tmp.y()-center.y(),2))) + center.x(), tmp.y()));
    }

    Circuit::self()->update();
}


void RotSwitch::updateStep()
{
    if( !m_needUpdate ) return;
    m_needUpdate = false;

    bool state;
    for (int n = 0; n < m_rows; n++) {
        if (m_dial->value() == n) {
            state = true;
        }
        else {
            state = false;
        }
        m_positions[n]->setSwitch(state);
        if (m_ways == 2)
            m_positions[n + m_rows]->setSwitch(state);
    }
    update();
}

double RotSwitch::rows()
{
    return m_rows;
}

void RotSwitch::setRows( double rows )
{
    m_rows = rows;
    if (m_ways == 1) {
      if (m_rows > 8) m_rows = 8;
      else if (m_rows < 3) m_rows = 3;
    }
    else {
      if (m_rows > 8) m_rows = 8;
      else if (m_rows < 3) m_rows = 3;
    }

    setupButtons();
    updateProxy();
}

int RotSwitch::ways()
{
    return m_ways;
}

void RotSwitch::setways( int ways )
{
    m_ways = ways;

    if (m_ways >= 2) {
     m_ways = 2;
     if (m_rows > 8) m_rows = 8;
    }
    else if (m_ways < 1) m_ways = 1;

    setupButtons();
}

void RotSwitch::updateProxy()
{
    m_proxy->setPos( QPoint(-m_dial->width()/2 - 32,
                            (-m_dial->height()/2)-((m_rows*8)/2)-4 ));
}

void RotSwitch::remove()
{
    for( MechContact* button : m_positions ) {
       m_positions.removeOne( button );
       Circuit::self()->removeComp( button );
    }

    Component::remove();
}

void RotSwitch::paint( QPainter *p, const QStyleOptionGraphicsItem *option, QWidget *widget )
{
    if( m_hidden ) return;

    Component::paint( p, option, widget );

    QPen pen = p->pen();
    pen.setWidth(2);
    p->setPen(pen);

    p->drawEllipse(center, 2, 2);
    p->drawLine( center, QPoint( -35, 0));

    for( int i=0; i < point_arc.length(); i++ ) {
        p->drawEllipse(point_arc.at(i), 1, 1);
        p->drawLine( point_arc.at(i), point_pin1.at(i));
    }
    p->drawLine(center, point_arc.at(m_dial->value()));

    if (m_ways == 2) {
        p->drawEllipse(center1, 2, 2);
        p->drawLine( center1, QPoint( -35, -(((m_rows-1)*8) +16)));

        for( int i=0; i < point_arc1.length(); i++ ) {
            p->drawEllipse(point_arc1.at(i), 1, 1);
            p->drawLine( point_arc1.at(i), point_pin2.at(i));
        }
        p->drawLine(center1, point_arc1.at(m_dial->value()));

        pen.setStyle( Qt::DashLine );
        pen.setWidth( 1 );
        p->setPen( pen );
        QLineF l1 = QLineF(center, point_arc.at(m_dial->value()));
        QLineF l2 = QLineF(center1, point_arc1.at(m_dial->value()));
        QPoint pt1 = QPoint(-16, l1.center().y());
        QPoint pt2 = QPoint(-16, l2.center().y());
        p->drawLine( l1.center(), pt1 );
        p->drawLine( l2.center(), pt2 );
        p->drawLine( pt1, pt2 );
    }
}
