/***************************************************************************
 *                                                                         *
 *   For this component I took inspiration from the PICSimLab simulator    *
 *   created by Luis Claudio Gambôa Lopes.                                 *
 *                                                                         *
 *   For Copyright see copyright.txt file at root folder                   *
 *                                                                         *
 ***************************************************************************/

#include <QGraphicsProxyWidget>
#include <QPainter>
#include <QFileInfo>
#include <QFileDialog>
#include <QDir>
#include <QAction>

#include "sd.h"
#include "simulator.h"
#include "itemlibrary.h"
#include "circuit.h"
#include "iopin.h"
#include "mainwindow.h"
#include "circuitwidget.h"
#include "utils.h"
#include "custombutton.h"

#include "propdialog.h"
#include "stringprop.h"

#define tr(str) simulideTr("Sd",str)

Component* Sd::construct( QString type, QString id )
{
    return new Sd( type, id );
}

LibraryItem* Sd::libraryItem()
{
    return new LibraryItem(
        tr("Sd Card Reader"),
        "Peripherals",
        "sd_close.png",
        "Sd",
        Sd::construct );
}

Sd::Sd( QString type, QString id )
       : Component( type, id )
       , eClockedDevice( id )
       , m_pinCS  ( 180, QPoint(-76, 12), id+"-PinCS"  , 0, this, input )
       , m_pinMosi( 180, QPoint(-76, -4), id+"-PinMosi" , 0, this, input )
       , m_pinSck ( 180, QPoint(-76, 4), id+"-PinSck" , 0, this, input )
       , m_pinMiso( 180, QPoint(-76,-12), id+"-PinMiso" , 0, this, output )
{
    m_pinCS.setLabelText( "CS" );
    m_pinCS.setInputHighV( 2.31 );
    m_pinCS.setInputLowV( 0.99 );

    m_pinMiso.setLabelText( "Miso" );
    m_pinMiso.setOutHighV( 3.0 );
    m_pinMiso.setOutLowV( 0.99 );
    m_pinMosi.setLabelText( "Mosi" );
    m_pinMosi.setInputHighV( 2.31 );
    m_pinMosi.setInputLowV( 0.99 );
    m_pinSck.setLabelText( "Sck" );
    m_pinSck.setInputHighV( 2.31 );
    m_pinSck.setInputLowV( 0.99 );

    m_pin.resize( 4 );
    m_pin[0] = &m_pinCS;
    m_pin[1] = &m_pinMosi;
    m_pin[2] = &m_pinSck;
    m_pin[3] = &m_pinMiso;
    for (int i=0; i < 4; i++) {
        m_pin[i]->setLabelColor(Qt::white);
    }
    m_clkPin = &m_pinSck;

    m_graphical = true;
    setShowId( false );
    m_area = QRectF( -68, -28, 100, 56 );
    m_background = ":/sd_open.png";
    open = true;
    setLabelPos(-70, -36, 0);

    m_button = new CustomButton();
    m_button->setMaximumSize( 8,8 );
    m_button->setGeometry(8,8,8,8);
    m_button->setCheckable( true );

    m_proxy = Circuit::self()->addWidget(m_button );
    m_proxy->setParentItem( this );
    m_proxy->setPos( QPoint(-60, 16) );

    m_keyEventConn = QObject::connect( m_button, &CustomButton::clicked , [=](){ onbuttonclicked(); });

    Simulator::self()->addToUpdateList( this );

    addPropGroup( { tr("Main"), {
        new StrProp <Sd>("File", tr("File"), ""
                             , this, &Sd::fileName, &Sd::setFile, 0 )
    },0} );
    
    sdcard_init();
}

Sd::~Sd(){QObject::disconnect( m_keyEventConn );}

void Sd::onbuttonclicked()
{
    if (!m_button->isChecked()) {
        m_background = ":/sd_open.png";
        open = true;
        sdcard_init();
    }
    else {
        if (!m_sd.fd) LoadFile();
        if (m_sd.card_present) {
           m_background = ":/sd_close.png";
           open = false;
        }
        else {
            open = true;
            m_button->setChecked(false);
            qDebug() << "SD Card ERROR";
        }
    }

    Circuit::self()->update();
}

void Sd::stamp()
{
    m_pinSck.changeCallBack( this ); // Register for Scl changes callback
    m_pinCS.changeCallBack( this );  // Register for CS changes callback

    m_pinMiso.scheduleState(false, 0);
}

void Sd::initialize()
{
    spi_insr = 0;           // m_rxReg
    spi_outsr = 0;          // m_txReg
    spi_bit = 0;
    spi_byte = 0;
}

void Sd::voltChanged()                    // Called when En Pin changes
{
    bool ret = false;

    if (m_sd.card_present == 0) return;         // Scheda non inserita

    if( m_pinCS.getInpState() ) {               // CS Pin High: SD not selected
        m_sd.replyc = 0;
        m_sd.data_rc = 0;
        m_sd.data_wc = 0;
        m_sd.multi_wr = 0;
        m_sd.multi_rd = 0;
        spi_insr = 0;           // m_rxReg
        spi_outsr = 0;          // m_txReg
        spi_bit = 0;
        spi_byte = 0;
        ret = true;
    }

    updateClock();

    if( m_clkState != Clock_Rising ){
        ret = true;                            // clock non in aumento
    }

    if( ret ) return;

    m_pinMiso.scheduleState((bool)(spi_outsr & 128), 0);      // Trasmette gli altri bit in spi_outsr

    spi_insr &= ~1; //Clear bit 0

    if( m_pinMosi.getInpState() ) spi_insr |= 1;              // Riceve 1 bit

    if( spi_bit == 7 )                                        // Se ho ricevuto 8 bit valuta comando
    {
        //qDebug() << "Rx: " << spi_insr;                       // spi_insr contiene byte ricevuto
        //qDebug() << "Tx: " << prev_txReg;
        spi_outsr = 0;
        proccessCommand();                                    // Comando
        //prev_txReg = spi_outsr;
        m_pinMiso.scheduleState((bool)(spi_outsr & 128), 0);  // Trasmette primo bit in spi_outsr
        spi_outsr <<= 1;
        spi_bit = 0;
        spi_byte++;
    }
    else
    {
        spi_insr <<= 1;
        spi_outsr <<= 1;
        spi_bit++;
    }
}

void Sd::proccessCommand() {

    unsigned int offset = 0;
    unsigned short crc16 = 0;
    QByteArray data;

    qDebug() << QString("sdcard byte in 0x%1  out 0x%2")
                .arg(spi_insr & 0xFF, 2, 16, QChar('0'))
                .arg((spi_outsr >> 8) & 0xFF, 2, 16, QChar('0'));

    if ((m_sd.data_wc) && (spi_byte == 1)) {
        spi_byte = 0;
        spi_outsr = (spi_outsr & 0xFF00) | 0xFF;
        if (m_sd.data_wc == 515)  // token
        {
            // must be 0xFE
            if (((spi_insr & 0xFF) != 0xFE) && ((spi_insr & 0xFF) != 0xFC)) {
                m_sd.data_wc = 516;
            }
        } else if (m_sd.data_wc < 3) {  // CRC
            if (m_sd.data_wc == 2) {
                m_sd.crc16 = (spi_insr & 0xFF) << 8;
            } else if (m_sd.data_wc == 1) {
                int resp = 0x05;
                m_sd.crc16 |= spi_insr & 0xFF;

                if (m_sd.crc_on) {
                    if (m_sd.crc16 != CRC16(buff, 512)) {
                        resp = 0x0B;  // Data rejected due to a CRC error
                    }
                }
                // write response
                spi_outsr = (spi_outsr & 0xFF00) | resp;
            }
        } else {
            buff[512 - (m_sd.data_wc - 2)] = spi_insr & 0xFF;
            qDebug() << QString("sdcard write buff[%1]= 0x%2")
                        .arg(512 - (m_sd.data_wc - 2))
                        .arg(spi_insr & 0xFF, 2, 16, QChar('0'));
            if ((m_sd.data_wc - 3) == 0) {
                m_sd.fd->write(reinterpret_cast<const char*>(buff), 512);
                qDebug() << QString("sdcard 512 bytes writed end %1").arg(m_sd.fd->pos() / 512);
            }
        }

        m_sd.data_wc--;

        if ((m_sd.multi_wr) && (!m_sd.data_wc)) {
            m_sd.data_wc = 515;
        }
    }
    else {
        switch (spi_byte << 3) {
        case 0:  // nothing
        case 8:  // command

            m_sd.cmd = spi_insr;
            if ((m_sd.cmd & 0xC0) == 0x40) {
                m_sd.cmd = spi_insr & 0x3F;
                m_sd.arg = 0;
                m_sd.cmd_buff[0] = spi_insr & 0xFF;
            } else {
                spi_outsr = 0xFF;
                spi_bit = 0;
                spi_byte = 0;
                m_sd.replyc = 0;
            }
            break;
        case 16:  // 1/4 parameter
            m_sd.arg |= (spi_insr & 0xFF) << 24;
            m_sd.cmd_buff[1] = spi_insr & 0xFF;
            break;
        case 24:  // 2/4 parameter
            m_sd.arg |= (spi_insr & 0xFF) << 16;
            m_sd.cmd_buff[2] = spi_insr & 0xFF;
            break;
        case 32:  // 3/4 parameter
            m_sd.arg |= (spi_insr & 0xFF) << 8;
            m_sd.cmd_buff[3] = spi_insr & 0xFF;
            break;
        case 40:  // 4/4parameter
            m_sd.arg |= (spi_insr & 0xFF);
            m_sd.cmd_buff[4] = spi_insr & 0xFF;
            break;
        case 48:  // crc
            m_sd.crc = (spi_insr & 0xFF);

            qDebug() << "----------------------------------------\n";

            qDebug() << QString("sdcard %1%2 [0x%3] arg=0x%4 CRC=0x%5")
                        .arg(m_sd.ap_cmd ? "ACMD" : "CMD")
                        .arg(m_sd.cmd)
                        .arg(m_sd.cmd, 2, 16, QChar('0'))
                        .arg(m_sd.arg, 8, 16, QChar('0'))
                        .arg(m_sd.crc, 2, 16, QChar('0'));

            if (m_sd.cmd_count < 10) {
                m_sd.cmd_count++;
                m_sd.R1 = 0x01;  // R1 |0|ParameterE|AddressE|EraseE|CRCE|IllegalC|EraseR|idle|
            } else {
                m_sd.R1 = 0x00;  // R1 |0|ParameterE|AddressE|EraseE|CRCE|IllegalC|EraseR|idle|
            }

            if (m_sd.crc_on) {
                if (m_sd.crc != ((CRC7(m_sd.cmd_buff, 5) << 1) | 0x01)) {
                    m_sd.R1 |= 0x80;       // R1 |0|ParameterE|AddressE|EraseE|CRCE|IllegalC|EraseR|idle|
                }
            }

            if (m_sd.ap_cmd) {
                switch (m_sd.cmd) {
                case ACMD13:                // SD_STATUS - Send the SD Status.
                    spi_outsr = 0xFF;         // fill byte
                    m_sd.reply[0] = m_sd.R1;
                    m_sd.reply[1] = 0xFE;     // start block
                    // SD statsdcard_t* us
                    memset(&m_sd.reply[2], 0, 64);
                    m_sd.reply[2] = 0x00;     //  DAT_BUS_WIDTH:2 SECURED_MODE:1 reserved:5
                    m_sd.reply[3] = 0X00;     //  reserved
                    m_sd.reply[4] = 0x00;
                    m_sd.reply[5] = 0x00;
                    m_sd.reply[6] = 0x00;
                    m_sd.reply[7] = 0x00;
                    m_sd.reply[8] = 0x00;
                    m_sd.reply[9] = 0x28;
                    m_sd.reply[10] = 0x01;
                    m_sd.reply[11] = 0x10;
                    m_sd.reply[12] = 0x90;
                    m_sd.reply[13] = 0x00;
                    m_sd.reply[14] = 0x0B;
                    m_sd.reply[15] = 0x05;
                    // reserved 16-65
                    m_sd.reply[66] = 0xbc;
                    m_sd.reply[67] = 0x65;
                    m_sd.replyc = 69;
                    qDebug() << "sdcard sd_status\n";
                    break;
                case ACMD23:  // SET_WR_BLK_ERASE_COUNT - Set the number of write blocks to be
                    // pre-erased before writing
                    spi_outsr = 0xFF;  // fill byte
                    m_sd.replyc = 2;
                    m_sd.reply[0] = m_sd.R1;
                    qDebug() << QString("sdcard pre-erased %1").arg(m_sd.arg);
                    break;
                case ACMD41:  // SD_SEND_OP_COMD - Sends host capacity support information and activates
                    // the card's initialization process
                    spi_outsr = 0xFF;  // fill byte
                    m_sd.replyc = 2;
                    m_sd.reply[0] = m_sd.R1;
                    qDebug() << "sdcard send_op_comd\n";
                    break;
                case ACMD42:  // SET_CLR_CARD_DETECT - Connect[1]/Disconnect[0] the 50KOhm pull-up
                    // resistor on CS (pin 1) of the card. The pull-up may be used for card
                    // detection.
                    spi_outsr = 0xFF;  // fill byte
                    m_sd.replyc = 2;
                    m_sd.reply[0] = m_sd.R1;
                    qDebug() << "sdcard set_clr_card_detect\n";
                    break;
                case ACMD51:              // SEND_SCR - Reads the SD Configuration Register (SCR).
                    spi_outsr = 0xFF;       // fill byte
                    m_sd.reply[0] = m_sd.R1;
                    m_sd.reply[1] = 0xFE;   // start block
                    // SCR
                    m_sd.reply[2] = 0x02;   // SCR_STRUCTURE:4 SD_SPEC :4
                    m_sd.reply[3] = 0X25;   // DATA_STAT_AFTER_ERASE:1 SD_SECURITY:3  SD_BUS_WIDTHS:4
                    m_sd.reply[4] = 0x80;   // reserved
                    m_sd.reply[5] = 0x00;   // reserved
                    m_sd.reply[6] = 0x00;   // reserved for manufacturer usage
                    m_sd.reply[7] = 0x00;   // reserved for manufacturer usage
                    m_sd.reply[8] = 0x00;   // reserved for manufacturer usage
                    m_sd.reply[9] = 0x00;   // reserved for manufacturer usage
                    m_sd.reply[10] = 0x4c;  // crc16
                    m_sd.reply[11] = 0xd7;
                    m_sd.replyc = 13;
                    qDebug() << "sdcard send_scr\n";
                    break;
                default:
                    spi_outsr = 0xFF;  // fill byte
                    m_sd.reply[0] =
                            0x04 | m_sd.R1;  // R1 |0|ParameterE|AddressE|EraseE|CRCE|IllegalC|EraseR|idle|
                    m_sd.replyc = 2;
                    qDebug() << QString("sdcard command ACMD%1 not implemented!!!").arg(m_sd.cmd);
                    break;
                }
                m_sd.ap_cmd = 0;
            } else {
                switch (m_sd.cmd) {
                case CMD0:            // GO_IDLE_STATE - init card in spi mode if CS low
                    spi_outsr = 0xFF;    // fill byte
                    m_sd.R1 = 1;
                    m_sd.reply[0] = m_sd.R1;
                    m_sd.replyc = 2;
                    m_sd.cmd_count = 0;
                    qDebug() << "sdcard soft reset \n";
                    break;
                case CMD1:                    // SEND_OP_COND
                    spi_outsr = 0xFF;  // fill byte
                    m_sd.reply[0] = m_sd.R1;
                    m_sd.replyc = 2;
                    qDebug() << QString("sdcard send op cond %1").arg(m_sd.arg);
                    break;
                case CMD8:  // SEND_IF_COND - verify SD Memory Card interface operating condition.
                    spi_outsr = 0xFF;  // fill byte
                    m_sd.replyc = 6;
                    m_sd.reply[0] = m_sd.R1;
                    m_sd.reply[1] = 0x00;
                    m_sd.reply[2] = 0x00;
                    m_sd.reply[3] = 0x01;             // voltage
                    m_sd.reply[4] = m_sd.arg & 0xFF;  // check patern
                    qDebug() << QString("sdcard send_if_cond %1").arg(m_sd.arg);
                    break;
                case CMD9:              // SEND_CSD - read the Card Specific Data (CSD register)
                    spi_outsr = 0xFF;      // fill byte
                    m_sd.reply[0] = m_sd.R1;
                    m_sd.reply[1] = 0xFE;  // start block
                    // CSDV2
                    m_sd.reply[2] = 0x40;  // csd_ver:2 reserved1:6
                    m_sd.reply[3] = 0X0E;  // taac:8 fixed 0x0E
                    m_sd.reply[4] = 0x00;  // nsac:8 fixed 0
                    m_sd.reply[5] = 0x5A;  // tran_speed:8
                    m_sd.reply[6] = 0x5B;  // ccc_high:8
                    m_sd.reply[7] = 0x59;  // ccc_low:4  read_bl_len:4   fixed to 0x09
                    m_sd.reply[8] = 0x00;  // read_bl_partial:1 write_blk_misalign:1  read_blk_misalign:1
                    // dsr_imp:1  reserved2:4 fixed to 0x00
                    m_sd.reply[9] =
                            (((m_sd.disk_size / 512) - 1) >> 16) & 0xFF;  // reserved3:2 c_size_high:6
                    m_sd.reply[10] = (((m_sd.disk_size / 512) - 1) >> 8) & 0xFF;  // c_size_mid:8
                    m_sd.reply[11] = ((m_sd.disk_size / 512) - 1) & 0xFF;         // c_size_low:8
                    m_sd.reply[12] = 0x7F;  // reserved4:1 erase_blk_en:1 sector_size_high:6
                    m_sd.reply[13] = 0x80;  // sector_size_low:1 wp_grp_size:7
                    m_sd.reply[14] =
                            0x0A;  // wp_grp_enable:1 reserved5:2  r2w_factor:3 write_bl_len_high:2
                    m_sd.reply[15] = 0x40;  // write_bl_len_low:2  write_partial:1 reserved6:5
                    m_sd.reply[16] = 0x00;  // file_format_grp:1  perm_write_protect:1 copy:1
                    // tmp_write_protect:1 file_format:2 reserved7:2
                    m_sd.reply[17] = (CRC7(&m_sd.reply[2], 15) << 1) | 1;  // crc:7 always1:1
                    crc16 = (CRC16(&m_sd.reply[2], 16));
                    m_sd.reply[18] = crc16 >> 8;
                    m_sd.reply[19] = crc16 & 0xFF;
                    m_sd.replyc = 21;
                    qDebug() << "sdcard send CSD\n";
                    break;
                case CMD10:  // SEND_CID - read the card identification information (CID register)
                    spi_outsr = 0xFF;  // fill byte
                    m_sd.reply[0] = m_sd.R1;
                    m_sd.reply[1] = 0xFE;  // start block
                    // CID
                    m_sd.reply[2] = 0x01;   // Manufacturer ID
                    m_sd.reply[3] = 0X34;   // OEM/Application high
                    m_sd.reply[4] = 0x32;   // OEM/Application low
                    m_sd.reply[5] = 'S';    // Product name
                    m_sd.reply[6] = 'D';    // Product name
                    m_sd.reply[7] = 'S';    // Product name
                    m_sd.reply[8] = 'I';    // Product name
                    m_sd.reply[9] = 'M';    // Product name
                    m_sd.reply[10] = 0x01;  //  prv_n:4 prv_m:4
                    m_sd.reply[11] = 0x12;  // psn
                    m_sd.reply[12] = 0x34;  // psn
                    m_sd.reply[13] = 0x56;  // psn
                    m_sd.reply[14] = 0x78;  // psn
                    m_sd.reply[15] = 0x00;  // reserved:4 mdt_year_high:4
                    m_sd.reply[16] = 0x04;  // mdt_year_low:4 mdt_month:4
                    m_sd.reply[17] = 0x03;  // crc:7 always1:1
                    m_sd.reply[18] = 0xfb;  // crc16
                    m_sd.reply[19] = 0x8b;
                    m_sd.replyc = 21;
                    qDebug() << "sdcard send CID \n";
                    break;
                case CMD12:                   // STOP_TRANSMISSION - end multiple block read sequence
                    spi_outsr = 0xFF;  // fill byte
                    m_sd.reply[0] = m_sd.R1;
                    m_sd.replyc = 2;
                    m_sd.multi_wr = 0;
                    m_sd.multi_rd = 0;
                    qDebug() << "sdcard stop transmission\n";
                    break;
                case CMD13:                   // SEND_STATUS - read the card status register
                    spi_outsr = 0xFF;  // fill byte
                    m_sd.reply[0] = m_sd.R1;
                    m_sd.reply[1] = 0x00;  // R2
                    m_sd.replyc = 3;
                    qDebug() << "sdcard send status\n";
                    break;
                case CMD16:                   // SET_BLOCKLEN - Sets a block length (in bytes) for all
                    spi_outsr = 0xFF;  // following block commands
                    m_sd.reply[0] = m_sd.R1;
                    m_sd.replyc = 2;
                    qDebug() << QString("sdcard set blocklen %1").arg(m_sd.arg);
                    break;
                case CMD17:  // READ_SINGLE_BLOCK - read a single data block from the card
                    spi_outsr = 0xFF;  // fill byte
                    m_sd.reply[0] = m_sd.R1;
                    m_sd.reply[1] = 0xFE;  // start block
                    m_sd.replyc = 3;
                    m_sd.data_rc = 512;
                    if (m_sd.fd->seek(m_sd.arg)) { qDebug() << "File seek successful at position" << m_sd.arg; }
                    else { qDebug() << "File seek failed!"; }
                    data = m_sd.fd->read(512);
                    memcpy(buff, data.data(), data.size());
                    qDebug() << QString("sdcard reading block %1").arg(m_sd.arg / 512);
                    break;
                case CMD18:  // READ_MULTIPLE_BLOCK - read a multiple data blocks from the card
                    spi_outsr = 0xFF;      // fill byte
                    m_sd.reply[0] = m_sd.R1;
                    m_sd.reply[1] = 0xFE;  // start block
                    m_sd.replyc = 3;
                    m_sd.data_rc = 512;
                    m_sd.multi_rd = 1;

                    if (m_sd.fd->seek(m_sd.arg)) { qDebug() << "File seek successful at position" << m_sd.arg; }
                    else { qDebug() << "File seek failed!"; }
                    data = m_sd.fd->read(512);
                    memcpy(buff, data.data(), data.size());
                    qDebug() << QString("sdcard reading multiple blocks start at %1").arg(m_sd.arg / 512);
                    break;
                case CMD24:          // WRITE_BLOCK - write a single data block to the card
                    spi_outsr = 0xFF;    // fill byte
                    m_sd.reply[0] = m_sd.R1;
                    m_sd.replyc = 2;
                    m_sd.data_wc = 515;  // include 0xFE initial token and crc
                    if (m_sd.fd->seek(m_sd.arg)) { qDebug() << "File seek successful at position" << m_sd.arg; }
                    else { qDebug() << "File seek failed!"; }
                    qDebug() << QString("sdcard writing block %1").arg(m_sd.arg / 512);
                    break;
                case CMD25:  // WRITE_MULTIPLE_BLOCK - write blocks of data until a STOP_TRANSMISSION
                    spi_outsr = 0xFF;  // fill byte
                    m_sd.reply[0] = m_sd.R1;
                    m_sd.replyc = 2;
                    m_sd.data_wc = 515;  // include 0xFC initial token and crc
                    m_sd.multi_wr = 1;
                    if (m_sd.fd->seek(m_sd.arg)) { qDebug() << "File seek successful at position" << m_sd.arg; }
                    else { qDebug() << "File seek failed!"; }
                    qDebug() << QString("sdcard writing multiple blocks start at %1").arg(m_sd.arg / 512);
                    break;
                case CMD32:  // ERASE_WR_BLK_START - sets the address of the first block to be erased
                    spi_outsr = 0xFF;  // fill byte
                    m_sd.reply[0] = m_sd.R1;
                    m_sd.replyc = 2;
                    m_sd.ebstart = m_sd.arg / 512;
                    qDebug() << QString("sdcard erase block start %1").arg(m_sd.ebstart);
                    break;
                case CMD33:  // ERASE_WR_BLK_END - sets the address of the last block of the continuous
                    // range to be erased
                    spi_outsr = 0xFF;  // fill byte
                    m_sd.reply[0] = m_sd.R1;
                    m_sd.replyc = 2;
                    m_sd.ebend = m_sd.arg / 512;
                    qDebug() << QString("sdcard erase block end  %1").arg(m_sd.ebend);
                    break;
                case CMD38:         // ERASE - erase all previously selected blocks
                    spi_outsr = 0xFF;  // fill byte
                    m_sd.reply[0] = m_sd.R1;
                    m_sd.replyc = 2;
                    qDebug() << QString("sdcard erasing blocks %1 to %2").arg(m_sd.ebstart).arg(m_sd.ebend);
                    // fseek(m_sd.fd, m_sd.ebstart, SEEK_SET);
                    if (m_sd.fd->seek(m_sd.arg)) { qDebug() << "File seek successful at position" << m_sd.arg; }
                    else { qDebug() << "File seek failed!"; }
                    memset(buff, 0, 512);
                    for (offset = m_sd.ebstart; offset < m_sd.ebend; offset++) {
                        if (m_sd.fd->isOpen()) { m_sd.fd->write(reinterpret_cast<const char*>(buff), 512); }
                    }
                    break;
                case CMD55:         // APP_CMD - escape for application specific command
                    spi_outsr = 0xFF;  // fill byte
                    m_sd.reply[0] = m_sd.R1;
                    m_sd.replyc = 2;
                    m_sd.ap_cmd = 1;
                    qDebug() << "sdcard app command\n";
                    break;
                case CMD58:         // READ_OCR - read the OCR register of a card
                    spi_outsr = 0xFF;  // fill byte
                    m_sd.replyc = 6;
                    m_sd.reply[0] = m_sd.R1;
                    if (m_sd.disk_size >= 4194304L) {
                        m_sd.reply[1] = 0xC1;  // Busy:1 CCS:1 resrved:5  S18A:1
                    } else {
                        m_sd.reply[1] = 0x81;  // Busy:1 CCS:1 resrved:5  S18A:1
                    }
                    m_sd.reply[2] = 0xFF;      // voltage window
                    m_sd.reply[3] = 0x80;      // voltage window:1 reseved:7
                    m_sd.reply[4] = 0x00;      // reseved:8
                    qDebug() << "sdcard read OCR \n";
                    break;
                case CMD59:                // CRC_ON_OFF - enable or disable CRC checking
                    spi_outsr = 0xFF;          // fill byte
                    m_sd.reply[0] = m_sd.R1;
                    m_sd.replyc = 2;
                    m_sd.crc_on = m_sd.arg & 0x01;
                    qDebug() << QString("sdcard CRC ON/OFF = %1").arg(m_sd.crc_on);
                    break;
                default:
                    // spi_bit = 0;
                    // spi_byte = 0;
                    spi_outsr = 0xFF;        // fill byte
                    m_sd.reply[0] =
                            0x04 | m_sd.R1;  // R1 |0|ParameterE|AddressE|EraseE|CRCE|IllegalC|EraseR|idle|
                    m_sd.replyc = 2;
                    qDebug() << QString("sdcard command CMD%1 not implemented!!!\n").arg(m_sd.cmd);
                    break;
                }
            }
            qDebug() << QString("Reply[0] = 0x%02X").arg(m_sd.reply[0]);
            break;
        default:  // data
            offset = ((spi_byte << 3) - 48) / 8;

            if (!m_sd.data_rc) {
                if (offset >= m_sd.replyc) {
                    spi_outsr = (spi_outsr & 0xFF00) | 0xFF;
                    spi_bit = 0;
                    spi_byte = 0;
                    if (m_sd.cmd == CMD0) {
                        sdcard_rst();
                    }
                } else {
                    spi_outsr = (spi_outsr & 0xFF00) | m_sd.reply[offset - 1];
                }
            } else {
                if (offset < m_sd.replyc) {
                    spi_outsr = (spi_outsr & 0xFF00) | m_sd.reply[offset - 1];
                } else {
                    if (m_sd.data_rc) {
                        spi_outsr = (spi_outsr & 0xFF00) | buff[512 - m_sd.data_rc];
                        m_sd.data_rc--;
                        if (!m_sd.data_rc) {
                            unsigned short crc16 = 0xFFFF;
                            if (m_sd.crc_on) {
                                crc16 = CRC16(buff, 512);
                            }
                            spi_byte = 6;
                            m_sd.reply[0] = crc16 >> 8;
                            m_sd.reply[1] = crc16 & 0xFF;
                            m_sd.replyc = 3;

                            if (m_sd.multi_rd) {
                                m_sd.reply[2] = 0xFF;
                                m_sd.reply[3] = 0xFE;  // start block
                                m_sd.replyc = 5;
                                m_sd.data_rc = 512;
                                m_sd.arg += 512;
                                data = m_sd.fd->read(512);
                                memcpy(buff, data.data(), data.size());
                                qDebug() << QString("sdcard reading next block at %1").arg(m_sd.arg / 512);
                            }
                        }
                    } else {
                        spi_outsr = (spi_outsr & 0xFF00) | crc16;  // crc
                        spi_bit = 0;
                        spi_byte = 0;
                    }
                }
            }

            qDebug() << QString("sdcard data offset=%03   out=0x%02  in=0x%02  cmd=0x%02")
                        .arg(offset)
                        .arg(spi_outsr >> 8)
                        .arg(spi_insr & 0xFF)
                        .arg(m_sd.cmd);
            break;
        }
    }
    qDebug() << "Tx: " << spi_outsr;
}


// ******************* Gestione file immagine disco *.img ***********************

void Sd::LoadFile()
{
    QString fil;
    if( fil.isEmpty() ) fil = Circuit::self()->getFilePath();

    const QString dir = fil;

    QString fileName = QFileDialog::getOpenFileName( 0l, tr("Load Disk Image"), dir,
                       tr("img files (*.img);;All files (*.*)"));

    if( fileName.isEmpty() ) return;      // User cancels loading
    if(!fileName.contains(".img")) return; // Bad file

    setFile( fileName );
}

void Sd::setFile( QString fileName )
{
    bool ret = false;
    QString msg = "";

    if( Simulator::self()->isRunning() ) CircuitWidget::self()->powerCircOff();

    if( fileName.isEmpty() || !QFile::exists(fileName)) return;

    QDir circuitDir;
    circuitDir = QFileInfo( Circuit::self()->getFilePath() ).absoluteDir();
    QString fileNameAbs  = circuitDir.absoluteFilePath( fileName );

    if( !QFileInfo::exists( fileNameAbs ) ) {
        msg = "Sd::setFile Error: file doesn't exist:\n"+fileNameAbs;
        ret = true;
    }

    m_Compsd = Circuit::self()->compList()->values();
    for( int i=0; i<m_Compsd.size(); ++i )              // Verify Sd istance file open
    {
        Component* comp = m_Compsd.at( i );
        if (comp->idLabel().contains("Sd-") && comp->idLabel() != this->idLabel()) {
            if (comp->getPropStr("File") == fileNameAbs) {
                msg = "Sd::setFile Error: file already in use:\n"+fileNameAbs;
                ret = true;
             }
        }
    }

    if (ret) {
        MessageBoxNB( tr("Error"), tr(msg.toLatin1()));
        return;
    }

    m_fileName = fileNameAbs;

    sdcard_set_filename(fileNameAbs);
}

void Sd::remove()
{
    Simulator::self()->remFromUpdateList( this );
    QObject::disconnect( m_keyEventConn );
    sdcard_end();
    Component::remove();
}

void Sd::paint( QPainter* p, const QStyleOptionGraphicsItem* option, QWidget* widget )
{
    Component::paint( p, option, widget );

    QPixmap m_image = QPixmap( m_background );
    p->drawPixmap( m_area, m_image, m_image.rect());

    p->setPen(QColor(240, 240, 240));
    QFont font=p->font();
    font.setPixelSize(8);
    p->setFont(font);
    p->rotate(90);
    int ox, oy;
    if (open) {ox = -20; oy = -24;}
    else {ox = -20; oy = -8;}
    p->drawText(ox, oy, 40, 9, Qt::AlignCenter, "SD card", nullptr);
}

// START ******************** SD CARD ******************************************

// Generator polynomial: G(x) = x7 + x3 + 1
unsigned char Sd::CRC7(const unsigned char* buffer, const unsigned int size) {
    const unsigned char poly = 0x89;
    unsigned char crc = 0;
    for (unsigned i = 0; i < size; i++) {
        crc ^= buffer[i];
        for (int j = 0; j < 8; j++) {
            crc = (crc & 0x80) ? ((crc ^ poly) << 1) : (crc << 1);
        }
    }
    return crc >> 1;
}

// Generator polynomial: x^16 + x^12 + x^5 + 1
unsigned short Sd::CRC16(const unsigned char* buffer, unsigned int size) {
    unsigned char x;
    unsigned short crc = 0;
    for (unsigned i = 0; i < size; i++) {
        x = crc >> 8 ^ buffer[i];
        x ^= x >> 4;
        crc = (crc << 8) ^ ((unsigned short)(x << 12)) ^ ((unsigned short)(x << 5)) ^ ((unsigned short)x);
    }
    return crc;
}

void Sd::sdcard_rst() {
    m_sd.cmd = 0;
    m_sd.arg = 0;
    m_sd.crc = 0;
    m_sd.replyc = 0;
    m_sd.ap_cmd = 0;
    m_sd.data_rc = 0;
    m_sd.data_wc = 0;
    m_sd.multi_wr = 0;
    m_sd.multi_rd = 0;
    m_sd.crc_on = 0;
    m_sd.R1 = 1;
    m_sd.cmd_count = 0;
    qDebug() << "rst sdcard\n";
}

void Sd::sdcard_init() {
    open = true;
    m_sd.card_present = 0;
    m_button->setChecked(false);
    sdcard_rst();
    m_sd.fd = NULL;
    m_sd.disk_size = 0;
    qDebug() << "init sdcard\n";
}

void Sd::sdcard_end() {
    if (m_sd.fd) {
        if (m_sd.fd->isOpen()) m_sd.fd->close();
    }
    sdcard_init();
}

void Sd::sdcard_set_card_present(unsigned char cp) {
    if (m_sd.fd) {
        if (m_sd.fd->isOpen()) { m_sd.card_present = cp; qDebug() << "CLOSE";}
        else  { m_sd.card_present = 0; qDebug() << "OPEN";}
    }
}

void Sd::sdcard_set_filename(QString& fname) {
    if (m_sd.fd) {
        if (m_sd.fd->isOpen()) m_sd.fd->close(); // Chiude il file precedente aperto
    }

    m_sd.fd = new QFile(fname);
    if (m_sd.fd->open(QIODevice::ReadWrite)) {
        QFileInfo fileInfo(*m_sd.fd);

        m_sd.disk_size = fileInfo.size() >> 10;  // Dimensione in KB

        qDebug() << "sdcard size=" << m_sd.disk_size << "KB  ->  " << (fileInfo.size() / 512) << "blocks";
    }
    else {
        qDebug() << "Failed to open file: " << fname;
        sdcard_end();
    }

    sdcard_set_card_present(1);
}

// END **************************** SD CARD ***********************************

