// Simulation de moteur pas à pas avec capacité
// de microstepping

#include "math.h"
#include "matrix.h"
#include "graphics.h"
#include "text.h"

#include "pulsefilter.h"
#include "dcmotor.h"

//---------------------------------------------------------
//
//---------------------------------------------------------

const array<array<double>> path = {{0.05, 0.05}, {0.23, 0.45}, {0.41, 0.05}, {0.59, 0.45}, {0.77, 0.05}, {0.95, 0.45}};

const array<double> segment_friction = { 0.2, 0.7, 0.5, 0.3, 0.9 };

const double TW_W = 0.040;
const double TW_L = 0.10;

const array<array<double>> trameway = {
	{TW_W/4, TW_L/2}, {TW_W/2, TW_L/2-TW_W/4}, {TW_W/2, -TW_L/2},
	{-TW_W/2, -TW_L/2}, {-TW_W/2, TW_L/2-TW_W/4}, {-TW_W/4, TW_L/2}
};
const array<array<double>> trameway2 = {
	{TW_W/2, -TW_L/2}, {-TW_W/2, TW_L/2-TW_W/4}, 
	{TW_W/2, TW_L/2-TW_W/4}, {-TW_W/2, -TW_L/2}
};

//---------------------------------------------------------
//
//---------------------------------------------------------

enum pinMode_t{
    undef_mode=0,
    input,
    openCo,
    output,
    source
};

const uint64 MILLISECOND = 1000000000;
const uint64 SECOND = 1000*MILLISECOND;

double angle = 0;
double t_last;

uint width = 640;
uint height = 320;
uint h_scale = 20;		// nombre d'unités pour width

IoPin@ START_pin = component.getPin("START");
IoPin@ END_pin = component.getPin("END");
IoPin@ S0_pin = component.getPin("S0");
IoPin@ S1_pin = component.getPin("S1");
IoPin@ S2_pin = component.getPin("S2");

DCMotor@ motor = DCMotor("A", "B", "E1");

double ratio = 0.05/(2*PI);

uint tramway_segment = 0;
double tramway_pos = 0.0;

//---------------------------------------------------------
//
//---------------------------------------------------------

void setNominal_voltage( double val ) {
	motor.voltage0 = val;
}

double getNominal_voltage() {
	return motor.voltage0;
}

void setNominal_speed( double val ) {
	motor.speed0 = val*PI/30;		// en rad/s
}

double getNominal_speed() {
	return motor.speed0*30/PI;		// en tr/min
}

void setResistance( double val ) {
	motor.resistance = val;
}

double getResistance() {
	return motor.resistance;
}

void setEncoder_pulses( uint val ) {
	motor.encoder_pulses = val;
}

uint getEncoder_pulses() {
	return motor.encoder_pulses;
}

void setRatio( double val ) {
	ratio = val;
}

double getRatio() {
	return ratio;
}

//---------------------------------------------------------
//
//----------------------------------------------------------


void setup() {
    print("Rotary coder init");
}

void reset() {
    print("resetting rotary coder");

	START_pin.setPinMode(output);
	END_pin.setPinMode(output);
	S0_pin.setPinMode(output);
	S1_pin.setPinMode(output);
	S2_pin.setPinMode(output);
	
	motor.reset();	
	
	tramway_segment = 0;
	tramway_pos = 0.0;
	
	component.addEvent(5*MILLISECOND);
	t_last = time();
}


void updateStep() {
	// update();
	draw();
}


void voltChanged() {
	// update();
}


void runEvent() {
	update();	
	component.addEvent(5*MILLISECOND);
}

//---------------------------------------------------------
//
//---------------------------------------------------------

void update() {
	double t = time();
	double dt = t - t_last;

	if ((tramway_segment==0) && (tramway_pos <= 0.001)) {
		START_pin.setVoltage(5.0);
	} else {
		START_pin.setVoltage(0.0);
	}
	if ((tramway_segment==(path.length()-2)) && (tramway_pos >= 0.999)) {
		END_pin.setVoltage(5.0);
	} else {
		END_pin.setVoltage(0.0);
	}

	S0_pin.setVoltage(((tramway_segment & 1)==0)? 0.0 : 5.0);
	S1_pin.setVoltage(((tramway_segment & 2)==0)? 0.0 : 5.0);
	S2_pin.setVoltage(((tramway_segment & 4)==0)? 0.0 : 5.0);
	
	double angle0 = motor.angle;
	motor.friction = segment_friction[tramway_segment];
	motor.update(t);
	moveTramway( ratio*angleDiff(motor.angle, angle0) );
	
	t_last = t;
}


void draw() {
	clear();
	//screen.setBackground(0xD0D0FF);
    fillRectangle(0,0,width,height,0xD0D0FF);

	drawGrid(h_scale, h_scale*height/width, 0x909090);
	drawPath();
	drawTramway();
}


void drawGrid( int nx, int ny, uint color ) {
	double x, y;
	for (int i=1; i<nx; i++) {
		x = width/nx*i;
		drawLine(x, 0, x, height, color);
	}
	for (int i=1; i<ny; i++) {
		y = height/ny*i;
		drawLine(0, y, width, y, color);
	}
}


void drawPath() {
	Matrix m_scale = getScaleMatrix();
	array<array<double>> p = polyMatrix(m_scale, path);

	uint lwidth = int(width/h_scale*0.25);

	double friction_max = 0;
	for (uint i=0; i<segment_friction.length(); i++) {
		if (segment_friction[i] > friction_max) 
			friction_max = segment_friction[i];
	}	
	
	for (uint i=1; i<p.length(); i++) {
		double f = 0.75*segment_friction[i-1]/friction_max;
		uint c = uint((0.24+f)*255);
		uint seg_color = colorInt(c, 0, 0);
		// drawFatPolygon(p, seg_color, false, 5);
		drawFatLine(p[i-1][0], p[i-1][1], p[i][0], p[i][1], seg_color, lwidth);
	}	
	
	int r = int(width/h_scale*0.4);
	fillCircle(int(p[0][0]), int(p[0][1]), r, 0x0000FF);
	
	int n = p.length();
	int x = int(p[n-1][0]);
	int y = int(p[n-1][1]);
	fillRectangle(x-r, y-r, x+r, y+r, 0x0000FF);
	
	//=== numéro des segments ===
	double h = width/h_scale*0.5;
	for (uint i=0; i<p.length()-1; i++) {
		array<double> p1 = getPathPos(path, i, 0.25);
		array<double> p2 = pointMatrix(m_scale, p1);
		drawText(int(p2[0])+h*0.75, int(p2[1]), h, 0xFF00FF, "seg"+i, 1);
	}
}


void drawTramway() {
	array<double> pos = getPathPos(path, tramway_segment, tramway_pos);
	Matrix m_translate = Translation2D(pos[0], pos[1]);
	
	double a = getPathOrient(path, tramway_segment);
	Matrix R = Rotation2D(a-PI/2);
	
	Matrix m_scale = getScaleMatrix();
	
	Matrix M = m_scale;
	M = M.opMul(m_translate);
	M = M.opMul(R);

	array<array<double>> p = polyMatrix(M, trameway);
	fillPolygon(p, 0xFFFFC0);		
	drawPolygon(p, 0x000000, true);		

	array<array<double>> p2 = polyMatrix(M, trameway2);
	drawPolygon(p2, 0x000000, false);		
}


Matrix getScaleMatrix( void ) {
	Matrix scale = Matrix(3, 3);
	scale.set(0, 0, width);
	scale.set(1, 1, width);
	return scale;
}

array<double> pointMatrix( Matrix M, array<double> &in p ) {
	// Appliquer une matrice à un point (produit de matrice)
	Vector v = Vector(p);
	Vector hv = v.ToHomogeneous();
	Vector HV = M.opMul(hv);
	return HV.FromHomogeneous();
}

array<array<double>> polyMatrix( Matrix M, array<array<double>> &in p ) {
	// Appliquer une matrice à un polygone
	array<array<double>> P;
	
	for (uint i=0; i<p.length(); i++) {
		array<double> V = pointMatrix(M, p[i]);
		P.insertLast(V);
	}
	return P;
}

array<double> getPathPos( array<array<double>> &in p, uint iseg, double rel_pos ) {
	double x1 = p[iseg][0];
	double y1 = p[iseg][1];
	double x2 = p[iseg+1][0];
	double y2 = p[iseg+1][1];
	double dx = x2 - x1;
	double dy = y2 - y1;
	array<double> res;
	res.insertLast(x1 + rel_pos*dx);
	res.insertLast(y1 + rel_pos*dy);
	return res;
}

double getPathOrient( array<array<double>> &in p, uint iseg ) {
	double x1 = p[iseg][0];
	double y1 = p[iseg][1];
	double x2 = p[iseg+1][0];
	double y2 = p[iseg+1][1];
	double dx = x2 - x1;
	double dy = y2 - y1;
	return arctan2(dy, dx);
}

double getPathSegLen( array<array<double>> &in p, uint iseg ) {
	double x1 = p[iseg][0];
	double y1 = p[iseg][1];
	double x2 = p[iseg+1][0];
	double y2 = p[iseg+1][1];
	double dx = x2 - x1;
	double dy = y2 - y1;
	return sqrt(dx*dx+dy*dy);
}

void moveTramway( double dl ) {
	double seg_l = getPathSegLen(path, tramway_segment);
	
	tramway_pos += dl/seg_l;
	
	if (tramway_pos > 1+0.001) {
		if (tramway_segment < path.length()-2) {
			tramway_segment += 1;
			// double seg2_l = getPathSegLen(path, tramway_segment);
			// tramway_pos = tramway_pos*seg2_l/seg_l - 1.001;
			tramway_pos = 0;
		} else {
			tramway_pos = 1.0;
		}
	} else 
	if (tramway_pos < -0.001) {
		if (tramway_segment > 0) {
			tramway_segment -= 1;
			// double seg2_l = getPathSegLen(path, tramway_segment);
			// tramway_pos = 1.0-0.001 + tramway_pos*seg2_l/seg_l;
			tramway_pos = 1;
		} else {	
			tramway_pos = 0;
		}
	}
}
