#!/usr/bin/env bash
set -euo pipefail

log() { printf "\n\033[1m%s\033[0m\n" "$*"; }
die() { printf "\n\033[31mERROR:\033[0m %s\n" "$*" >&2; exit 1; }

MXE_DIR="${MXE_DIR:-$HOME/mxe}"
TARGET="${TARGET:-x86_64-w64-mingw32.static}"
JOBS="${JOBS:-$(nproc)}"
QT_PKGS="${QT_PKGS:-qtbase qtsvg qtserialport qtmultimedia qttools}"
CLEAN_QT="${CLEAN_QT:-0}"

log "0) Infos ressources (WSL2)"
echo "CPU: $(nproc)"
free -h || true
swapon --show || true

log "1) Détection OS"
. /etc/os-release
echo "OS: $NAME ($ID) $VERSION_ID"

log "2) Installation prérequis (apt)"
sudo apt update

# Base packages common to Debian/Ubuntu
BASE_PKGS=(
  build-essential git make wget curl unzip xz-utils p7zip-full
  autoconf automake autopoint libtool libtool-bin bison flex gperf
  gettext intltool pkg-config
  python3 python3-mako
  ruby perl
  cmake meson ninja-build
  libssl-dev zlib1g-dev libpcre2-dev
  libgdk-pixbuf2.0-bin
  ccache
  lzip
)

# python-is-python3 is Ubuntu-native; on Debian it's optional/available depending on release.
# We'll try install it but ignore failure.
sudo apt install -y "${BASE_PKGS[@]}"

if ! command -v python >/dev/null 2>&1; then
  log "python (alias) absent -> tentative d'installation python-is-python3"
  sudo apt install -y python-is-python3 || true
fi
command -v python >/dev/null 2>&1 || echo "NOTE: 'python' non présent. (MXE accepte python3, mais certains scripts préfèrent python.)"

log "3) Clone / update MXE"
if [[ -d "$MXE_DIR/.git" ]]; then
  cd "$MXE_DIR"
  git fetch --all --prune
  if git show-ref --verify --quiet refs/remotes/origin/master; then
    git reset --hard origin/master
  elif git show-ref --verify --quiet refs/remotes/origin/main; then
    git reset --hard origin/main
  else
    die "Ni origin/master ni origin/main trouvés."
  fi
else
  git clone https://github.com/mxe/mxe.git "$MXE_DIR"
  cd "$MXE_DIR"
fi

log "4) settings.mk"
cat > "$MXE_DIR/settings.mk" <<EOF
MXE_TARGETS := ${TARGET}
JOBS := \$(shell nproc)
MXE_USE_CCACHE := mxe
MXE_MESON_OPTS += --wrap-mode=nodownload
EOF
sed -n '1,80p' "$MXE_DIR/settings.mk"

if [[ "$CLEAN_QT" == "1" ]]; then
  log "5) Nettoyage ciblé Qt"
  rm -rf "$MXE_DIR/usr/${TARGET}/qt5" || true
  rm -rf "$MXE_DIR"/tmp-qtbase-* "$MXE_DIR"/tmp-qtsvg-* "$MXE_DIR"/tmp-qtserialport-* \
         "$MXE_DIR"/tmp-qtmultimedia-* "$MXE_DIR"/tmp-qttools-* || true
fi

log "6) Build toolchain + Qt"
cd "$MXE_DIR"
make -j"$JOBS" cc
make -j"$JOBS" $QT_PKGS

log "7) Vérifs Qt + lrelease"
QMAKE="$MXE_DIR/usr/bin/${TARGET}-qmake-qt5"
[[ -x "$QMAKE" ]] || die "qmake introuvable: $QMAKE"
"$QMAKE" -v

QT_VER="$("$QMAKE" -query QT_VERSION || true)"
[[ "$QT_VER" == 5.15.* ]] || die "Qt n'est pas en 5.15.x (QT_VERSION=$QT_VER)."

export PATH="$MXE_DIR/usr/bin:$MXE_DIR/usr/${TARGET}/qt5/bin:$PATH"
command -v lrelease >/dev/null 2>&1 || die "lrelease introuvable (qttools manquant ?)"
echo "OK: lrelease = $(command -v lrelease)"
lrelease -version || true

log "OK: MXE + Qt prêts."
