#!/usr/bin/env bash
set -euo pipefail

# ============================================================
# build_simulide_mxe.sh
# - Build SimulIDE-dev for Windows (x86_64-w64-mingw32.static) using MXE Qt5
# - Idempotent: can be re-run; cleans build dir only (not whole MXE).
# ============================================================

# --------- User config ---------
MXE_DIR="${MXE_DIR:-$HOME/mxe}"
TARGET="${TARGET:-x86_64-w64-mingw32.static}"

SRC_ROOT="${SRC_ROOT:-$HOME/src}"
REPO_URL="${REPO_URL:-https://github.com/eeTools/SimulIDE-dev.git}"
REPO_DIR="${REPO_DIR:-$SRC_ROOT/SimulIDE-dev}"

BUILD_DIR="${BUILD_DIR:-$REPO_DIR/build_win64_static}"
PRO_FILE_REL="${PRO_FILE_REL:-build_XX/SimulIDE_Build.pro}"

# If you want a fully clean rebuild each run:
CLEAN_BUILD="${CLEAN_BUILD:-1}"   # 1=yes, 0=no

# --------- Derived paths ---------
QMAKE="$MXE_DIR/usr/bin/${TARGET}-qmake-qt5"
CC="$MXE_DIR/usr/bin/${TARGET}-gcc"
CXX="$MXE_DIR/usr/bin/${TARGET}-g++"
AR="$MXE_DIR/usr/bin/${TARGET}-ar"
RANLIB="$MXE_DIR/usr/bin/${TARGET}-ranlib"
STRIP="$MXE_DIR/usr/bin/${TARGET}-strip"
OBJDUMP="$MXE_DIR/usr/bin/${TARGET}-objdump"

QT_BIN_DIR="$MXE_DIR/usr/${TARGET}/qt5/bin"
MXE_BIN_DIR="$MXE_DIR/usr/bin"

# --------- Helpers ---------
log() { printf "\n\033[1m%s\033[0m\n" "$*"; }
die() { printf "\n\033[31mERROR:\033[0m %s\n" "$*" >&2; exit 1; }

need_file() {
  local f="$1"
  [[ -f "$f" ]] || die "Fichier introuvable: $f"
}

need_exe() {
  local f="$1"
  [[ -x "$f" ]] || die "Exécutable introuvable/non exécutable: $f"
}

# --------- 0) Sanity checks ---------
log "0) Vérifs de base (MXE, toolchain, Qt)"

need_exe "$QMAKE"
need_exe "$CC"
need_exe "$CXX"
need_exe "$AR"
need_exe "$RANLIB"
need_exe "$STRIP"
need_exe "$OBJDUMP"

log "Qt version (qmake -v) :"
"$QMAKE" -v

# Check expected Qt 5.15.x
QT_VER="$("$QMAKE" -query QT_VERSION || true)"
[[ "$QT_VER" == 5.15.* ]] || die "Qt n'est pas en 5.15.x (QT_VERSION=$QT_VER)."

# Check settings.mk contains Meson nodownload (if file exists)
if [[ -f "$MXE_DIR/settings.mk" ]]; then
  log "Vérif settings.mk : Meson wrap-mode=nodownload"
  if ! grep -q -- '--wrap-mode=nodownload' "$MXE_DIR/settings.mk"; then
    echo "NOTE: '$MXE_DIR/settings.mk' ne contient pas --wrap-mode=nodownload."
    echo "      (ce n'est pas bloquant pour SimulIDE, mais utile pour glib/meson wraps)"
  else
    echo "OK: --wrap-mode=nodownload trouvé dans settings.mk"
  fi
else
  echo "NOTE: $MXE_DIR/settings.mk absent (pas bloquant)."
fi

# Ensure lrelease is available (qttools)
log "Vérif lrelease (Qt Linguist Tools)"
export PATH="$MXE_BIN_DIR:$QT_BIN_DIR:$PATH"

if ! command -v lrelease >/dev/null 2>&1; then
  # Try to locate any lrelease in MXE
  FOUND_LRELEASE="$(find "$MXE_DIR/usr" -type f \( -name lrelease -o -name lrelease.exe -o -name "*lrelease*" \) 2>/dev/null | head -n 1 || true)"
  if [[ -n "${FOUND_LRELEASE:-}" ]]; then
    echo "Trouvé un lrelease candidat : $FOUND_LRELEASE"
    echo "Ajoute son dossier au PATH si nécessaire."
    export PATH="$(dirname "$FOUND_LRELEASE"):$PATH"
  fi
fi

command -v lrelease >/dev/null 2>&1 || die "lrelease introuvable. Il faut construire qttools côté MXE : cd ~/mxe && make qttools"
echo "OK: lrelease = $(command -v lrelease)"
lrelease -version || true

# --------- 1) Clone / update SimulIDE ---------
log "1) Clone / update SimulIDE-dev"

mkdir -p "$SRC_ROOT"
if [[ -d "$REPO_DIR/.git" ]]; then
  log "Repo déjà présent -> git fetch + reset sur origin/master (ou main si besoin)"
  cd "$REPO_DIR"
  git fetch --all --prune
  # Essaie master puis main
  if git show-ref --verify --quiet refs/remotes/origin/master; then
    git reset --hard origin/master
  elif git show-ref --verify --quiet refs/remotes/origin/main; then
    git reset --hard origin/main
  else
    die "Ni origin/master ni origin/main trouvés."
  fi
else
  cd "$SRC_ROOT"
  git clone "$REPO_URL" "$REPO_DIR"
  cd "$REPO_DIR"
fi

# --------- 2) Check .pro exists ---------
log "2) Vérif du .pro de build"
PRO_FILE="$REPO_DIR/$PRO_FILE_REL"
need_file "$PRO_FILE"
echo "OK: .pro = $PRO_FILE"

# --------- 3) Prepare build dir ---------
log "3) Préparation du build out-of-tree"
mkdir -p "$BUILD_DIR"
cd "$BUILD_DIR"

if [[ "$CLEAN_BUILD" == "1" ]]; then
  log "Nettoyage idempotent du build dir (sans toucher aux sources)"
  rm -rf .qmake.stash Makefile* build/ release/ debug/ object_script.* moc_* ui_* rcc_* 2>/dev/null || true
  # Certains projets créent build_win64_static/build/... :
  rm -rf build 2>/dev/null || true
fi

# --------- 4) Generate Makefile with qmake ---------
log "4) qmake (génération Makefile) + environnement MXE"
export CC="${TARGET}-gcc"
export CXX="${TARGET}-g++"
export AR="${TARGET}-ar"
export RANLIB="${TARGET}-ranlib"
export STRIP="${TARGET}-strip"
export PKG_CONFIG="${TARGET}-pkg-config"

echo "PATH=$PATH"
echo "CC=$CC"
echo "CXX=$CXX"
echo "QMAKE=$QMAKE"

"$QMAKE" "$PRO_FILE" CONFIG+=release

# Quick sanity: ensure Makefile exists
[[ -f Makefile ]] || die "Makefile non généré par qmake."

log "Makefile: CC/CXX détectés"
grep -nE '^(CC|CXX)\s*=' Makefile | head -n 20 || true

# --------- 5) Build ---------
log "5) Compilation (make)"
make -j"$(nproc)"

# --------- 6) Locate exe (search build dir + repo outputs) ---------
log "6) Recherche de l'exécutable .exe (build dir + build_XX/executables)"

# Cherche d'abord dans build dir, puis dans build_XX/executables, puis partout dans le repo
CANDIDATES=()
while IFS= read -r f; do CANDIDATES+=("$f"); done < <(find "$BUILD_DIR" -type f -iname "*.exe" 2>/dev/null || true)
while IFS= read -r f; do CANDIDATES+=("$f"); done < <(find "$REPO_DIR/build_XX/executables" -type f -iname "*.exe" 2>/dev/null || true)
while IFS= read -r f; do CANDIDATES+=("$f"); done < <(find "$REPO_DIR" -type f -iname "*.exe" 2>/dev/null || true)

# Dédupliquer
UNIQ=()
declare -A seen
for f in "${CANDIDATES[@]}"; do
  [[ -n "${seen["$f"]+x}" ]] && continue
  seen["$f"]=1
  UNIQ+=("$f")
done

[[ ${#UNIQ[@]} -gt 0 ]] || die "Aucun .exe trouvé dans $BUILD_DIR ni dans $REPO_DIR"

# Choisir le plus récent (mtime)
EXE="$(ls -1t "${UNIQ[@]}" | head -n 1)"
echo "OK: EXE = $EXE"

# --------- 7) Inspect DLL deps ---------
log "7) Dépendances DLL (objdump -p | DLL Name)"
"$OBJDUMP" -p "$EXE" | grep -i 'DLL Name' || true

log "Terminé."
