// Simulation de moteur pas à pas avec capacité
// de microstepping

#include "graphics.h"

IoPin@ A1_pin = component.getPin("A1");
IoPin@ B1_pin = component.getPin("B1");
IoPin@ A2_pin = component.getPin("A2");
IoPin@ B2_pin = component.getPin("B2");
IoPin@ com_pin = component.getPin("com");


class PulseFilter {
	double output, t_last, a, v_last;
	
	PulseFilter() {
		output = 0;
		v_last = 0;
		a = 10.0;
		t_last = time();
	}
	
	double f( double vin ) {
		double t = time();
		output += (v_last-output)*(t-t_last)*a;
		t_last = t;
		v_last = vin;
		return output;
	}
}

PulseFilter A1();
PulseFilter B1();
PulseFilter A2();
PulseFilter B2();


double fc = 500.0;
double fe = 5000.0;

int width = 256;
int height = 256;

uint poles = 4;
bool bipolar = false;
double inertia = 10;
double friction = 2;
uint linked_value = 1000;

double angle = 0;
double speed = 0;
double t_last;

void setPoles( uint val ) {
	poles = val;
	if (poles < 1) poles = 1;
}

uint getPoles() {
	return poles;
}

void setInertia( double val ) {
	inertia = val;
	if (inertia <= 0) inertia = 0.000000001;
}

double getInertia() {
	return inertia;
}

void setFriction( double val ) {
	friction = val;
}

double getFriction() {
	return friction;
}

void setBipolar( bool val ) {
	bipolar = val;
}

bool getBipolar() {
	return bipolar;
}

void setLinked_value( uint val ) {
	linked_value = val;
}

uint getLinked_value() {
	return linked_value;
}

void setup() {
    print("microstepper init");
	
	print("number of poles : "+poles);
}

void reset() {
    print("resetting microstepper");

    screen.setBackground(0xF0FFF0);
	
    A1_pin.setPinMode( 1 ); // Input
	B1_pin.setPinMode( 1 ); // Input
	A2_pin.setPinMode( 1 ); // Input
	B2_pin.setPinMode( 1 ); // Input
	com_pin.setPinMode( 1 ); // Input	
	
	angle = 0.0;
	t_last = time();
	
    A1_pin.changeCallBack( element, true );
    A2_pin.changeCallBack( element, true );
    B1_pin.changeCallBack( element, true );
    B2_pin.changeCallBack( element, true );
    com_pin.changeCallBack( element, true );
}


void updateStep() {
	draw();
	//update();

	double a = angleDiff(angle, 0);
	double p2 = 2.0*PI;
	double a2 = (a<0)? p2+a:a;
	uint ia = uint(linked_value*a2/p2);
	component.setLinkedValue(0, double(ia), 0);
	component.setLinkedString(1, ""+(a*180.0/PI), 0);
}


void voltChanged() {
	update();
}


void update() {
	string debug = "";
	double dt = time() - t_last;

	array<double> v;
	double com = com_pin.getVoltage();
	double va1, va2, vb1, vb2;
	if (bipolar) {
		double dv = A1_pin.getVoltage()-A2_pin.getVoltage();
		va1 = dv/2;
		va2 = -dv/2;
		dv = B1_pin.getVoltage()-B2_pin.getVoltage();
		vb1 = dv/2;
		vb2 = -dv/2;
	} else {
		va1 = A1_pin.getVoltage()-com;
		vb1 = B1_pin.getVoltage()-com;
		va2 = A2_pin.getVoltage()-com;
		vb2 = B2_pin.getVoltage()-com;
	}
	v.insertLast(A1.f(va1));
	v.insertLast(B1.f(vb1));
	v.insertLast(A2.f(va2));
	v.insertLast(B2.f(vb2));	

	double da = 2.0*PI/(4*poles);
	double a1, a2;

	array<double> a;
	array<double> va;
	
	// Déterminer les poles
	double rg;
	if (bipolar) {
		rg = 1.0;
	} else {
		rg = (poles > 1)? 2.1 : 1.9;
	}	
	a1 = angle - da*rg;
	a2 = angle + da*rg;
	int ia=ceil(a1/da); 
	while (ia <= a2/da) {
		a.insertLast(ia*da);
		va.insertLast(v[modulo4(ia)]);
		ia++;
	}
	
	// Calculer le baricentre (pole dominant)
	double torque = 0;
	double bc = 0;
	double w = 0;
	for (uint i=0; i<a.length(); i++) {
		bc += va[i]*a[i];
		w += va[i];
	}

	// Si poids non nul, alors pole dominant
	if (w > 0.0000000001) {
		bc = bc / w;

		// Dessiner un repere du baricentre
		//double x = width/2+width*0.25*cos(bc);
		//double y = height/2-height*0.25*sin(bc);
		//drawLine(width/2,height/2,x,y,0xFF0000);

		// Se deplacer vers pole dominant
		torque = (bc-angle)*w;
	}

	double acc = torque/inertia;
	speed += (acc-friction*speed)*dt;
	angle += speed*dt;
	
	t_last = time();
}


int modulo4( int i ) {
	while (i < 0) i += 4;
	return i % 4;
}


double angleDiff( double a, double b ) {
	// différence d'angles -PI à +PI
    double diff = a - b;
    while (diff <= -PI) diff += 2.0 * PI;
    while (diff > PI) diff -= 2.0 * PI;
    return diff;
}


void draw() {
	clear();
	screen.setBackground(0x7070A0);

	double x0 = width/2;
	double y0 = height/2;

	double rr = width/2*0.85;
	double rr2 = rr*1.1;
	double rs = width/2*0.8;
	
	drawStator(x0, y0, rr, rr2, poles, 0xFFFFFF);
	drawRotor(x0, y0, rs, 0xFFFF00);
}


void drawStatorOnePole( double xc, double yc, double r1, double r2, double a1, double a2, uint color ) {
	double da = (a2-a1)/16;
	int n = 3;
	for (int i=0; i<16; i++) {
		double r = (n < 2)? r2 : r1;
		drawArc(xc, yc, r, a1+i*da, a1+(i+1)*da, color);
		if ((n == 0) || (n == 2)) {
			double c = cos(a1+i*da);
			double s = sin(a1+i*da);
			double x1 = xc+r1*c;
			double y1 = yc-r1*s;
			double x2 = xc+r2*c;
			double y2 = yc-r2*s;
			drawLine(x1, y1, x2, y2, color);
		}
		n = (n+1)%4;
	}	
}


void drawStator( double xc, double yc, double r1, double r2, int N, uint color ) {
	double da = PI*2/N;
	for (double a=0; a<PI*2; a+=da) {
		drawStatorOnePole(xc, yc, r1, r2, a, a+da, color);
	}
}


void drawRotor( double xc, double yc, double r, uint color ) {
	drawArc(xc, yc, r, 0, PI, color);
	drawArc(xc, yc, r, PI, PI*2.0, color);
	drawArc(xc, yc, r/6, 0, PI, color);
	drawArc(xc, yc, r/6, PI, PI*2.0, color);
	double x1 = xc + r*cos(angle);
	double y1 = yc - r*sin(angle);
	double x2 = xc + r*cos(angle+PI+20*PI/180);
	double y2 = yc - r*sin(angle+PI+20*PI/180);
	double x3 = xc + r*cos(angle+PI-20*PI/180);
	double y3 = yc - r*sin(angle+PI-20*PI/180);
	drawLine(x1, y1, x2, y2, color);
	drawLine(x2, y2, x3, y3, color);
	drawLine(x3, y3, x1, y1, color);
}


