// Simulation de moteur pas à pas avec capacité
// de microstepping

IoPin@ A1_pin = component.getPin("A1");
IoPin@ B1_pin = component.getPin("B1");
IoPin@ A2_pin = component.getPin("A2");
IoPin@ B2_pin = component.getPin("B2");
IoPin@ com_pin = component.getPin("com");


class PulseFilter {
	double output, t_last, a, v_last;
	
	PulseFilter() {
		output = 0;
		v_last = 0;
		a = 10.0;
		t_last = time();
	}
	
	double f( double vin ) {
		double t = time();
		output += (v_last-output)*(t-t_last)*a;
		t_last = t;
		v_last = vin;
		return output;
	}
}

PulseFilter A1();
PulseFilter B1();
PulseFilter A2();
PulseFilter B2();


double fc = 500.0;
double fe = 5000.0;

int width = 256;
int height = 256;

int poles = 4;
bool bipolar = false;
double inertia = 10;
double friction = 2;

double angle = 0;
double speed = 0;
double t_last;

void setPoles( int val ) {
	poles = val;
	if (poles < 0) poles = 1;
}

int getPoles() {
	return poles;
}

void setInertia( double val ) {
	inertia = val;
	if (inertia <= 0) inertia = 0.000000001;
}

double getInertia() {
	return inertia;
}

void setFriction( double val ) {
	friction = val;
}

double getFriction() {
	return friction;
}

void setBipolar( bool val ) {
	bipolar = val;
}

bool getBipolar() {
	return bipolar;
}

void setup() {
    print("microstepper init");
	
	print("number of poles : "+poles);
}

void reset() {
    print("resetting microstepper");

    screen.setBackground(0xF0FFF0);
	
    A1_pin.setPinMode( 1 ); // Input
	B1_pin.setPinMode( 1 ); // Input
	A2_pin.setPinMode( 1 ); // Input
	B2_pin.setPinMode( 1 ); // Input
	com_pin.setPinMode( 1 ); // Input	
	
	angle = 0.0;
	t_last = time();
	
    A1_pin.changeCallBack( element, true );
    A2_pin.changeCallBack( element, true );
    B1_pin.changeCallBack( element, true );
    B2_pin.changeCallBack( element, true );
    com_pin.changeCallBack( element, true );
}


double time() {
	uint64 ctime = component.circTime();
	return ctime/1000000000000.0;
}


void updateStep() {
	draw();
	//update();
}


void voltChanged() {
	update();
}


void update() {
	string debug = "";
	double dt = time() - t_last;

	array<double> v;
	double com = com_pin.getVoltage();
	// v.insertLast(A1_pin.getVoltage()-com);
	// v.insertLast(B1_pin.getVoltage()-com);
	// v.insertLast(A2_pin.getVoltage()-com);
	// v.insertLast(B2_pin.getVoltage()-com);
	v.insertLast(A1.f(A1_pin.getVoltage()-com));
	v.insertLast(B1.f(B1_pin.getVoltage()-com));
	v.insertLast(A2.f(A2_pin.getVoltage()-com));
	v.insertLast(B2.f(B2_pin.getVoltage()-com));	

	double da = 2.0*PI/(4*poles);
	double a1, a2;

	array<double> a;
	array<double> va;
	
	// Déterminer les poles
	double rg = (poles > 1)? 2.1 : 1.9;
	a1 = angle - da*rg;
	a2 = angle + da*rg;
	int ia=ceil(a1/da); 
	while (ia <= a2/da) {
		a.insertLast(ia*da);
		va.insertLast(v[modulo4(ia)]);
		ia++;
	}
	
	// Calculer le baricentre (pole dominant)
	double torque = 0;
	double bc = 0;
	double w = 0;
	for (uint i=0; i<a.length(); i++) {
		bc += va[i]*a[i];
		w += va[i];
	}

	// Si poids non nul, alors pole dominant
	if (w > 0.0000000001) {
		bc = bc / w;

		// Dessiner un repere du baricentre
		//double x = width/2+width*0.25*cos(bc);
		//double y = height/2-height*0.25*sin(bc);
		//drawLine(width/2,height/2,x,y,0xFF0000);

		// Se deplacer vers pole dominant
		torque = (bc-angle)*w;
	}

	double acc = torque/inertia;
	speed += (acc-friction*speed)*dt;
	angle += speed*dt;
	
	t_last = time();
}


int modulo4( int i ) {
	while (i < 0) i += 4;
	return i % 4;
}


double angleDiff( double a, double b ) {
	// différence d'angles -PI à +PI
    double diff = a - b;
    while (diff <= -PI) diff += 2.0 * PI;
    while (diff > PI) diff -= 2.0 * PI;
    return diff;
}


void clear() {
    screen.setBackground( 0 );
}


void draw() {
	clear();
	screen.setBackground(0x7070A0);

	double x0 = width/2;
	double y0 = height/2;

	double rr = width/2*0.85;
	double rr2 = rr*1.1;
	double rs = width/2*0.8;
	
	drawStator(x0, y0, rr, rr2, poles, 0xFFFFFF);
	drawRotor(x0, y0, rs, 0xFFFF00);
}


void drawStatorOnePole( double xc, double yc, double r1, double r2, double a1, double a2, uint color ) {
	double da = (a2-a1)/16;
	int n = 3;
	for (int i=0; i<16; i++) {
		double r = (n < 2)? r2 : r1;
		drawArc(xc, yc, r, a1+i*da, a1+(i+1)*da, color);
		if ((n == 0) || (n == 2)) {
			double c = cos(a1+i*da);
			double s = sin(a1+i*da);
			double x1 = xc+r1*c;
			double y1 = yc-r1*s;
			double x2 = xc+r2*c;
			double y2 = yc-r2*s;
			drawLine(x1, y1, x2, y2, color);
		}
		n = (n+1)%4;
	}	
}


void drawStator( double xc, double yc, double r1, double r2, int N, uint color ) {
	double da = PI*2/N;
	for (double a=0; a<PI*2; a+=da) {
		drawStatorOnePole(xc, yc, r1, r2, a, a+da, color);
	}
}


void drawRotor( double xc, double yc, double r, uint color ) {
	drawArc(xc, yc, r, 0, PI, color);
	drawArc(xc, yc, r, PI, PI*2.0, color);
	drawArc(xc, yc, r/6, 0, PI, color);
	drawArc(xc, yc, r/6, PI, PI*2.0, color);
	double x1 = xc + r*cos(angle);
	double y1 = yc - r*sin(angle);
	double x2 = xc + r*cos(angle+PI+20*PI/180);
	double y2 = yc - r*sin(angle+PI+20*PI/180);
	double x3 = xc + r*cos(angle+PI-20*PI/180);
	double y3 = yc - r*sin(angle+PI-20*PI/180);
	drawLine(x1, y1, x2, y2, color);
	drawLine(x2, y2, x3, y3, color);
	drawLine(x3, y3, x1, y1, color);
}


//---------------------------------------------------------
//--- MATH FUNCTIONS --------------------------------------
//---------------------------------------------------------

const double PI = 3.14159265358979323846;

int abs( int n ) {
	return (n >= 0)? n : -n;
}

double abs(double x) {
    return x < 0.0 ? -x : x;
}

double fmod(double a, double b) {
	return a % b;
}

// Implémentation de floor si elle n'est pas disponible
int floor(double x) {
    if (x >= 0) {
        return int(x);
    } else {
        return int(x) - 1;
    }
}

// Implémentation de ceil si elle n'est pas disponible
int ceil(double x) {
    if (x == int(x)) {
        return int(x);
    } else if (x > 0) {
        return int(x) + 1;
    } else {
        return int(x);
    }
}

double sin(double x) {
    // Constantes
    const double TWO_PI = 2*PI;

    // Réduction de l'angle
    x = x - TWO_PI * int(x / TWO_PI);
    if (x > PI) x -= TWO_PI;
    if (x < -PI) x += TWO_PI;

    // Série de Taylor pour sin(x)
    double term = x;
    double sum = x;
    double x2 = x * x;

    term *= -x2 / (2 * 3);
    sum += term;

    term *= -x2 / (4 * 5);
    sum += term;

    term *= -x2 / (6 * 7);
    sum += term;

    term *= -x2 / (8 * 9);
    sum += term;

    return sum;
}

double cos(double x) {
    const double TWO_PI = 2*PI;

    x = x - TWO_PI * int(x / TWO_PI);
    if (x > PI) x -= TWO_PI;
    if (x < -PI) x += TWO_PI;

    double term = 1.0;
    double sum = 1.0;
    double x2 = x * x;

    term *= -x2 / (1 * 2);
    sum += term;

    term *= -x2 / (3 * 4);
    sum += term;

    term *= -x2 / (5 * 6);
    sum += term;

    term *= -x2 / (7 * 8);
    sum += term;

    return sum;
}

double sqrt(double x) {
    if (x < 0.0) {
        return 0.0;
    }

    double guess = x / 2.0;
    const double epsilon = 0.00001;

    while (abs(guess * guess - x) > epsilon) {
        guess = (guess + x / guess) / 2.0;
    }

    return guess;
}

float log(float x) {
    if (x <= 0.0) {
        // Gérer les entrées invalides
        // Ici, nous retournons 0.0, mais vous pouvez gérer différemment
        return 0.0;
    }

    // Transformation pour améliorer la convergence
    float y = (x - 1.0) / (x + 1.0);
    float y_squared = y * y;

    float sum = 0.0;
    float term = y; // Premier terme y^1 / 1
    int n = 1;

    const int MAX_TERMS = 20;      // Nombre maximal de termes
    const float EPSILON = 0.00001; // Précision souhaitée

    while (n <= MAX_TERMS) {
        sum += term / n;
        term *= y_squared; // y^{2k+1}
        n += 2;

        if (abs(term / n) < EPSILON) {
            break; // Arrêt si la contribution est négligeable
        }
    }

    return 2.0 * sum;
}

float exp(float x) {
    float sum = 1.0;        // Term 0: 1
    float term = 1.0;       // Initial term
    int n = 1;              // Term index

    const int MAX_TERMS = 20;      // Nombre maximal de termes
    const float EPSILON = 0.00001; // Précision souhaitée

    while (n < MAX_TERMS) {
        term *= x / n;       // Calcul du terme suivant: x^n / n!
        sum += term;         // Ajout du terme à la somme

        if (abs(term) < EPSILON) {
            break;            // Arrêt si la contribution est négligeable
        }

        n++;
    }

    return sum;
}


//---------------------------------------------------------
//--- DRAW FUNCTIONS --------------------------------------
//---------------------------------------------------------

void drawLine(double x0, double y0, double x1, double y1, uint color) {
	//=== Tracer une ligne ===
	
	int x = int(x0);
	int y = int(y0);
	int xend = int(x1);
	int yend = int(y1);

    int dx = abs(xend - x);
    int dy = abs(yend - y);

    int sx = x0 < x1 ? 1 : -1;
    int sy = y0 < y1 ? 1 : -1;

    int err = dx - dy;
	
    while (true) {
        screen.setPixel(x, y, color);

        if (x == xend && y == yend)
            break;

        int e2 = 2 * err;

        if (e2 > -dy) {
            err -= dy;
            x += sx;
        }

        if (e2 < dx) {
            err += dx;
            y += sy;
        }
    }
}


void drawPolygon(array<array<double>> &in polygone, uint color, bool close) {
	//=== Tracer un polygone ===
	
    uint n = polygone.length();
    if (n < 2) return; // Besoin d'au moins deux points pour tracer des lignes

    for (uint i = 1; i < n; i++) {
		double x0 = polygone[i-1][0];
		double y0 = polygone[i-1][1];
        double x1 = polygone[i][0]; // Boucle vers le premier sommet
        double y1 = polygone[i][1];

        drawLine(int(x0), int(y0), int(x1), int(y1), color);
    }
	if (close) {
		double x0 = polygone[n-1][0];
		double y0 = polygone[n-1][1];
        double x1 = polygone[0][0]; // Boucle vers le premier sommet
        double y1 = polygone[0][1];
		drawLine(int(x0), int(y0), int(x1), int(y1), color);
	}
}


void drawArc(double xc, double yc, double radius, double startAngle, double endAngle, uint color) {
	//=== Tracer un arc de cercle ===
    
    array<array<double>> arcPoints;
	const double PI2 = PI*2.0;

	startAngle = fmod(startAngle, PI2);
	endAngle = fmod(endAngle, PI2);
	
	while (endAngle < startAngle) endAngle += PI2;
	double len = (endAngle-startAngle)*radius;
	int n = ceil(len / 5.0);
	double da = (endAngle-startAngle)/n;
	
	for (double a=startAngle; a <= endAngle+da/5; a+=da) {
		double x = xc + radius * cos(a);
		double y = yc - radius * sin(a);
		arcPoints.insertLast({x, y});
		//print(""+a+":"+x+","+y);
	}

    // Tracer l'arc en tant que polygone
    drawPolygon(arcPoints, color, false);
}
